// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   utility/options/Option.hh
/// @brief  Program option interface class
/// @author Stuart G. Mentzer (Stuart_Mentzer@objexx.com)


#ifndef INCLUDED_utility_options_Option_HH
#define INCLUDED_utility_options_Option_HH


// Unit headers
#include <utility/options/Option.fwd.hh>

// Package headers
#include <utility/options/keys/OptionKey.hh>

// C++ headers
#include <cstddef>
#include <string>


namespace utility {
namespace options {


/// @brief Program option interface class
class Option
{


protected: // Types


	/// @brief Option state enumeration
	enum State {
		INACTIVE, // No default or user-specified value
		DEFAULT,  // Default value active
		USER      // User-specified value active
	};


public: // Types


	// STL/boost style
	typedef  OptionKey  key_type;
	typedef  std::size_t  size_type;

	// Project style
	typedef  OptionKey  Key;
	typedef  std::size_t  Size;


protected: // Creation


	/// @brief Default constructor
	inline
	Option()
	{}


	/// @brief Copy constructor
	inline
	Option( Option const & )
	{}


public: // Creation


	/// @brief Clone this
	virtual
	Option *
	clone() const = 0;


	/// @brief Destructor
	inline
	virtual
	~Option()
	{}


protected: // Assignment


	/// @brief Copy assignment
	inline
	Option &
	operator =( Option const & )
	{
		return *this;
	}


public: // Methods


	/// @brief Activate
	virtual
	Option &
	activate() = 0;


	/// @brief Deactivate
	virtual
	Option &
	deactivate() = 0;


	/// @brief Set to default value, if any
	virtual
	Option &
	to_default() = 0;


	/// @brief Clear
	virtual
	Option &
	clear() = 0;


	/// @brief Value assignment from a command line string
	virtual
	Option &
	cl_value( std::string const & value_str ) = 0;


	/// @brief Legal specifications check: Report and return error state
	virtual
	bool
	legal_specs_report() const = 0;


	/// @brief Legal value limits check: Report and return error state
	virtual
	bool
	legal_limits_report() const = 0;


	/// @brief Legal size limits check: Report and return error state
	virtual
	bool
	legal_size_report() const = 0;


	/// @brief Legal default value check: Report and return error state
	virtual
	bool
	legal_default_report() const = 0;


	/// @brief Legal default value check
	virtual
	void
	legal_default_check() const = 0;


	/// @brief Legal value check: Report and return error state
	virtual
	bool
	legal_report() const = 0;


	/// @brief Legal value check
	virtual
	void
	legal_check() const = 0;


	/// @brief Required specified option check: Report and return error state
	virtual
	bool
	specified_report() const = 0;


	/// @brief Required specified option check
	virtual
	void
	specified_check() const = 0;


public: // Properties


	/// @brief Key
	virtual
	Key const &
	key() const = 0;


	/// @brief ID
	virtual
	std::string const &
	id() const = 0;


	/// @brief Identifier
	virtual
	std::string const &
	identifier() const = 0;


	/// @brief Code
	virtual
	std::string const &
	code() const = 0;


	/// @brief Name
	virtual
	std::string const &
	name() const = 0;


	/// @brief Description
	virtual
	std::string const &
	description() const = 0;


	/// @brief Legal or inactive default value?
	virtual
	bool
	legal_default() const = 0;


	/// @brief Legal value?
	virtual
	bool
	legal() const = 0;


	/// @brief Has a default?
	virtual
	bool
	has_default() const = 0;


	/// @brief Default active?
	virtual
	bool
	default_active() const = 0;


	/// @brief Default inactive?
	virtual
	bool
	default_inactive() const = 0;


	/// @brief Active?
	virtual
	bool
	active() const = 0;


	/// @brief Inactive?
	virtual
	bool
	inactive() const = 0;


	/// @brief Defaulted?
	virtual
	bool
	defaulted() const = 0;


	/// @brief User-specified?
	virtual
	bool
	user() const = 0;


	/// @brief Specified (user-specified)?
	virtual
	bool
	specified() const = 0;


	/// @brief Present (user-specified)?
	virtual
	bool
	present() const = 0;


	/// @brief Is a string readable as this option's value type?
	virtual
	bool
	is_value( std::string const & value_str ) const = 0;


	/// @brief Is a string readable as this option's value type and a legal command line value?
	virtual
	bool
	is_cl_value( std::string const & value_str ) const = 0;


	/// @brief Can another value be added and stay within any size constraints?
	virtual
	bool
	can_hold_another() const = 0;


	/// @brief Default size (number of default values)
	virtual
	Size
	default_size() const = 0;


	/// @brief Number of default values (default size)
	virtual
	Size
	n_default_value() const = 0;


	/// @brief Size (number of values)
	virtual
	Size
	size() const = 0;


	/// @brief Number of values (size)
	virtual
	Size
	n_value() const = 0;


	/// @brief Option type code string representation
	virtual
	std::string
	type_string() const = 0;


	/// @brief Legal value string representation
	virtual
	std::string
	legal_string() const = 0;


	/// @brief Size constraint string representation
	virtual
	std::string
	size_constraint_string() const = 0;


	/// @brief Default value string representation
	virtual
	std::string
	default_string() const = 0;


	/// @brief Value string representation
	virtual
	std::string
	value_string() const = 0;


	/// @brief =Value string representation
	virtual
	std::string
	equals_string() const = 0;


public: // Comparison


	/// @brief Option < Option
	/// @note  Key-based ordering
	/// @note  Needed for use as option in associative containers
	friend
	inline
	bool
	operator <( Option const & a, Option const & b )
	{
		return ( a.key() < b.key() );
	}


}; // Option


} // namespace options
} // namespace utility


#endif // INCLUDED_utility_options_Option_HH
