// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 23432 $
//  $Date: 2008-06-24 16:25:52 +0300 (Tue, 24 Jun 2008) $
//  $Author: yab $


// Rosetta Headers
#include "ssblocks.h"
#include "after_opts.h"
#include "fold_abinitio.h"
#include "force_barcode.h"
#include "fragments.h"
#include "fragments_ns.h"
#include "misc.h"
#include "monte_carlo.h" // yab: misc removal
#include "param.h"
#include "random_numbers.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/formatted.o.hh>

// C++ Headers
#include <algorithm>
#include <iostream>

//namespaces

namespace block_stuff {
	int nseg;
	FArray1D_int begin_cons_block( 30 );
	FArray1D_int end_cons_block( 30 );
	FArray1D_float block_avg_conf( 30 );
}

namespace block_for_eval {
	using namespace param;
	FArray1D_int ss_state( MAX_RES() );
	FArray1D_char consHE( MAX_RES(), ' ' );
}

//db identify_ssblocks() tabulates the secondary structure state of each
//db position in the fragment files (currently it uses the first position
//db in each 9 residue fragment).  residues which are either "H" or "E" more than
//db 70% of the time are identified; conf_sectruct(i) is set to true,
//db and consHE(i) to "H" or "E".
//db Then, blocks of consective residues where conf_secstruct(i) is true are
//db identified; the beginning and end of each block are stored in
//db begin_cons_block and end_cons_block and nseg is the total number of
//db blocks.  the frequency of the "majority"
//db secondary structure in each block is recorded in block_avg_confidence.

//db decide_block_state() is called at the beginning of each independent
//db simulation to determine whether each block is in the "majority" or
//db "minority" state (block_state(i) is set to either true or false
//db for each block i).  the frequency with which  any given block is set to
//db true is simply the block_avg_confidence. the integer array
//db  ss_state(total_residue) is set to 0 for residues not in blocks, to
//db  1 for residues in "minority" (block_state(i)=false) blocks, or to
//db  2 for residues in "majority" blocks.  to include cases where the
//db correct answer is neither all majority or all minority, each block
//db is ignored 20% of the time (ss_state is set to 0 for block residue)

//db  scoreall_frag_block() is not currently used--it scores each
//db fragment's consistency with the current block states and stores the
//db results in frag_block_score.  (this is a reasonably big array)

//db block_rank_frags calls score_frag_block which builds an array of
//db pointers to fragments which are consistent with the current block
//db states (ie, if a given block is in its minority, false state, only
//db fragments which do not contain the majority secondary structure at
//db any of the regions which overlap with the block are incorporated into
//db the allowed fragment list for that position. the list of pointers
//db produced by the two functions is block_frag_pointer(nn_num,res_num,size_bin)
//db and the number of allowed fragments at each position is in
//db block_depth(res_num,size_bin).
//db the current version of score_frag_block uses a cems heap sort to retrieve not only the fragments
//db which are perfectly consistent with the current block state, but also
//db those that have less than a specified number of mismatches.  I
//db decided just to set a minimum (5) and maximum (25) number of fragments at
//db each position and pick the fragments from the top of the fragment
//db lists that were perfectly consistent with the block states instead of
//db specifying a particular number of mismatches.
//db
//db   choose_fragment is changed to pick from the sets of allowed
//db fragments rather than the set of all fragments using the
//db block_depth and block_frag_pointer arrays.


////////////////////////////////////////////////////////////////////////////////
/// @begin get_ssblock_state
///
/// @brief
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
get_ssblock_state()
{
	static bool ssblock_state = { false };
	static bool init = { false };

	if ( !init ) {
		ssblock_state = truefalseoption("ssblocks");
		init = true;
	}
	return ssblock_state;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin identify_ssblocks
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
identify_ssblocks()
{
	using namespace block_for_eval;
	using namespace block_stuff;
	using namespace fragments;
	using namespace misc;
	using namespace param;

	int bin1;
	int pos;
	FArray2D_float frag_secstruct( MAX_RES()(), 3 );
	FArray1D_int consensus_secstruct( MAX_RES()() );
	FArray1D_bool conf_secstruct( MAX_RES()() );


	if ( !get_ssblock_state()|| barcode_exist() ) return;

	for ( int i = 1; i <= total_residue; ++i ) {
		consensus_secstruct(i) = 0;
		consHE(i) = ' ';
		frag_secstruct(i,1) = 0;
		frag_secstruct(i,2) = 0;
	}

	bin1 = get_index_by_frag_size(9);
	for ( int i = 1; i <= total_residue; ++i ) {
		if ( i > total_residue-8 ) {
			pos = total_residue-8;
		} else {
			pos = i;
		}

		for ( int j = 1, je = align_depth(pos,bin1); j <= je; ++j ) {
			if ( ss_type(pos,j,i-pos,bin1) == 'H' ) {
				++frag_secstruct(i,1);
			} else {
				if ( ss_type(pos,j,i-pos,bin1) == 'E' ) {
					++frag_secstruct(i,2);
				} else {
					++frag_secstruct(i,3);
				}
			}
		}
		for ( int j = 1; j <= 3; ++j ) {
			frag_secstruct(i,j) /= align_depth(pos,bin1);
		}
	}
	for ( int i = 1; i <= total_residue; ++i ) {
		conf_secstruct(i) = false;
		for ( int j = 1; j <= 2; ++j ) { // we are not interested (?) in loop blocks
			if ( frag_secstruct(i,j) > 0.7 ) {
				conf_secstruct(i) = true;
				consensus_secstruct(i) = j;
				if ( j == 1 ) {
					consHE(i) = 'H';
				} else {
					consHE(i) = 'E';
				}
			}
		}
	}
	nseg = 0;
	if ( conf_secstruct(1) ) {
		++nseg;
		begin_cons_block(nseg) = 1;
	}
	for ( int i = 2; i <= total_residue; ++i ) {
		if ( conf_secstruct(i) && !conf_secstruct(i-1) ) {
			++nseg;
			begin_cons_block(nseg) = i;
		}

		if ( conf_secstruct(i-1) && !conf_secstruct(i) ) {
			end_cons_block(nseg) = i-1;

// db how long should blocks be?  try allowing even 1 residue blocks
//     if ( end_cons_block(nseg)-begin_cons_block(nseg) < 1 ) --nseg;

		}
	}
	if ( conf_secstruct(total_residue) ) {
		end_cons_block(nseg) = total_residue;
//     if ( end_cons_block(nseg)-begin_cons_block(nseg) < 1 ) --nseg;
	}

// to allow some play at the ends of long blocks, shorten them slightly

	for ( int i = 1; i <= nseg; ++i ) {
		if ( end_cons_block(i) - begin_cons_block(i) > 7 ) {
			++begin_cons_block(i);
			--end_cons_block(i);
		}
	}


	for ( int i = 1; i <= nseg; ++i ) {
		block_avg_conf(i) = 0.;
		for ( int j = begin_cons_block(i), je = end_cons_block(i); j <= je; ++j ) {
			block_avg_conf(i) += frag_secstruct(j,consensus_secstruct(j));
		}
		block_avg_conf(i) /= (end_cons_block(i)-begin_cons_block(i)+1);
	}

	for ( int i = 1; i <= total_residue; ++i ) {
		std::cout << SS( i );
		for ( int j = 1; j <= 3; ++j ) {
			std::cout << SS( frag_secstruct(i,j) );
		}
		std::cout  << consensus_secstruct(i) << conf_secstruct(i) << std::endl;
	}

	for ( int i = 1; i <= nseg; ++i ) {
		std::cout << SS( i ) << SS( begin_cons_block(i) ) <<
		 SS( end_cons_block(i) ) << SS( block_avg_conf(i) ) << std::endl;
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin decide_block_state
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
decide_block_state()
{
	using namespace block_for_eval;
	using namespace block_stuff;
	using namespace misc;

//car local
	FArray1D_bool block_state( 30 );


	if ( !get_ssblock_state() ) return;
  if (!barcode_exist()) {
	for ( int i = 1; i <= nseg; ++i ) {
		if ( ran3() > block_avg_conf(i) ) {
			block_state(i) = false;
		} else {
			block_state(i) = true;
		}
	}

	for ( int i = 1; i <= total_residue; ++i ) {
		ss_state(i) = 0;
	}

	std::cout << "block_state" << std::endl;
	for ( int i = 1; i <= nseg; ++i ) {
		if ( ran3() > 0.2 ) {
			std::cout << SS( i ) << SS( block_state(i) ) << std::endl;
			for ( int j = begin_cons_block(i), je = end_cons_block(i); j <= je; ++j ) {
				if ( !block_state(i) ) {
					ss_state(j) = 1;
				} else {
					ss_state(j) = 2;
				}
			}
		} else {
			std::cout << SS( i ) << SS( block_state(i) ) << "off" << std::endl;
		}
	}
	}

//db  the next two calls are for diagnostics only--they
//db  list the scores for the top 25 fragments according
//db to the current block states

//       scoreall_frag_block(9);
//       scoreall_frag_block(3);

//db   build sorted lists of fragments according to consistency with
//db     blocks

	block_rank_frags(9);
	block_rank_frags(3);
	block_rank_frags(1);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin block_rank_frags
///
/// @brief
///
/// @detailed
///
/// @param  size - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
block_rank_frags( int size )
{
	using namespace fragments::frag_pointer;
	using namespace misc;


	int size_bin = get_index_by_frag_size(size);

	if ( barcode_exist() ) {
		for ( int i = 1, e = total_residue-size + 1; i <= e; ++i ) {
			barcode_score_frag_block_heap(size,i,block_frag_pointer(1,i,size_bin),
			 block_depth(i,size_bin));
		}
	} else {
		for ( int i = 1, e = total_residue-size + 1; i <= e; ++i ) {
			score_frag_block_heap(size,i,block_frag_pointer(1,i,size_bin),
			block_depth(i,size_bin));
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin score_frag_block_heap
///
/// @brief
///db this function  utilizes heap to allow selection of
///db mostly compatible fragments (0.0 < block_score < 0.x)
///
/// @detailed
///
/// @param  size - [in/out]? - insertion position (first res in frag)
/// @param  ires - [in/out]? - size is 3 or 9 usually
/// @param  frag_block_pointer - [in/out]? - output array of pointers to frags
///        ordered by our scheme for ranking them using pointers
/// @param  block_frag_depth - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
score_frag_block_heap(
	int & size,
	int & ires,
	FArray1Da_int frag_block_pointer,
	int & block_frag_depth
)
{
	using namespace block_for_eval;
	using namespace fragments;
	using namespace param;

// frag list is a list of fragments at a given insertion position
// max_frag is the physical size of the array

	frag_block_pointer.dimension( MAX_NEIGH() );

	FArray1D_int heap( DRange( -1, MAX_NEIGH()() ) ); // local storage for the sort
	FArray1D_float coheap( MAX_NEIGH()() );
	float mismatch_frag_block;
	bool err;
	float num;

	float block_score;
	float dummy;

	int size_bin = get_index_by_frag_size(size);
	heap_init(heap,coheap,25);
	int count = 0;
	for ( int j = 1, je = align_depth(ires,size_bin); j <= je; ++j ) {
		mismatch_frag_block = 0.;
		num = 0.;
		block_score = 0.0;
		for ( int k = 0; k < size; ++k ) {
			if ( ss_state(ires+k) > 0 ) {
				++num;
				if ( ss_state(ires+k) == 1 &&
				 consHE(ires+k) == ss_type(ires,j,k,size_bin) )
				 mismatch_frag_block += 0.66;
				if ( ss_state(ires+k) == 2 &&
				 consHE(ires+k) != ss_type(ires,j,k,size_bin) )
				 mismatch_frag_block += 1.0;
			}
		}
		if ( num > 0.0 ) {
			block_score = mismatch_frag_block/num;
		}
		if ( block_score < 0.0001 ) {
			++count;
			frag_block_pointer(count) = j;
		} else {
			heap_insert(heap,coheap,j,-block_score,err);
		}

	}                     // end j (depth)

	block_frag_depth = count;
	block_score = 0.;
	for ( int j = count + 1, je = std::min( 5, align_depth(ires,size_bin) ); j <= je; ++j ) {
		heap_extract(heap,coheap,frag_block_pointer(j),dummy,err);
		++block_frag_depth;
	}

	block_frag_depth = std::min( get_number9merfrags() ,block_frag_depth);

}
////////////////////////////////////////////////////////////////////////////////
/// @begin barcode_score_frag_block_heap
///
/// @brief
///db this function  utilizes heap to allow selection of
///db mostly compatible fragments (0.0 < block_score < 0.x)
///
/// @detailed
///
/// @param  size - [in/out]? - insertion position (first res in frag)
/// @param  ires - [in/out]? - size is 3 or 9 usually
/// @param  frag_block_pointer - [in/out]? - output array of pointers to frags
///        ordered by our scheme for ranking them using pointers
/// @param  block_frag_depth - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
barcode_score_frag_block_heap(
	int & size,
	int & ires,
	FArray1Da_int frag_block_pointer,
	int & block_frag_depth
)
{
	using namespace block_for_eval;
	using namespace fragments;
	using namespace param;

// frag list is a list of fragments at a given insertion position
// max_frag is the physical size of the array

	frag_block_pointer.dimension( MAX_NEIGH() );

	FArray1D_int heap( DRange( -1, MAX_NEIGH()() ) ); // local storage for the sort
	FArray1D_float coheap( MAX_NEIGH()() );
	float mismatch_frag_block;
	bool err;
	float num;

	float block_score, min_score, max_score;
	int size_bin = get_index_by_frag_size(size);
	heap_init(heap,coheap,200);

	min_score = 0.0;
	max_score = 0.0;
	for ( int k = 0; k < size; ++k ) {
		min_score += min_barcode_score_residue( ires+k );
		max_score += max_barcode_score_residue( ires+k );
	}

	for ( int j = 1, je = align_depth(ires,size_bin); j <= je; ++j ) {
		mismatch_frag_block = 0.;
		num = 0.;
		block_score = 0.0;
		for ( int k = 0; k < size; ++k ) {
			block_score += barcode_score_residue( ires+k, align_phi(ires,j,k,size_bin),
			 align_psi(ires,j,k,size_bin), align_omega(ires,j,k,size_bin), ss_type(ires,j,k,size_bin) );
		}

		heap_insert( heap, coheap, j, block_score, err );
	}                     // end j (depth)

	// The higher the score, the more constraints violated
	block_frag_depth = 0;
	int max_depth = std::min(align_depth(ires,size_bin), 
			(size == 9) ? get_number9merfrags() : get_number3merfrags());
	int dummy_block_pointer;
	int num_good_frags = 0;
	// The logic here is a little convoluted.  The idea is to add at least 5
	// fragments for every position.  min_score is the best barcode score we
	// can get for this window; add all fragments that satisfy it.  These best
	// fragments will be diluted by a factor of at most 2 by the addition of
	// fragments which satisfy some barcodes but not all.  max_score
	// is the worst barcode score we can get for this window; add no fragments
	// with this score unless needed to reach the bare minimum quota of 5.  
	for ( int j = 1; j <= max_depth; ++j ) {
		heap_extract( heap, coheap, dummy_block_pointer, block_score, err );
		if(block_score > min_score) {
			if(j == 1) { // if NO fragments have minimum score, redefine it.  
				min_score = block_score;
				++num_good_frags;
			} else if(j > 5) {
				if(block_score == max_score)
					break; // If we're already into the worst possible fragments
				if(j > num_good_frags * 2 )
					break; // If we've diluted the best fragments by a factor of 2
			}
		} else 
			++num_good_frags;
		frag_block_pointer(j) = dummy_block_pointer;
		++block_frag_depth;
	}
	std::cout << "  Size: " << SS(size) <<  "  NUMBER OF FRAGS FOR POS: " << SS(ires) << SS(block_frag_depth) << std::endl;
}

//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//db diagnostic functions
////////////////////////////////////////////////////////////////////////////////
/// @begin output_block_state
///
/// @brief
///
/// @detailed
///
/// @param  iunit - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
output_block_state( std::ostream & iunit )
{
	using namespace block_for_eval;
	using namespace block_stuff;

	if ( !get_ssblock_state() || barcode_exist() ) return;

	iunit << "block_summary" <<
	 ' ' << F( 7, 3, mc_global_track::mc_score::low_score ) <<
	 ' ' << F( 7, 3, mc_global_track::diagnose::low_rms );
	for ( int k = 1; k <= nseg; ++k ) {
		iunit << I( 2, ss_state(begin_cons_block(k)) );
	}
	iunit << std::endl;

}

//////////////////////////////////////////////////////////////////////////////
/// @begin mismatch_frag_block
///
/// @brief
///
/// @detailed
///
/// @param  frag_begin - [in/out]? -
/// @param  size - [in/out]? -
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
mismatch_frag_block(
	int const frag_begin,
	int const size
)
{
	using namespace block_for_eval;
	using namespace misc;

	float mismatch_frag_block; // Return value

	float frac, num;


	mismatch_frag_block = 0;
	num = 0;
	for ( int i = frag_begin, e = frag_begin + size - 1; i <= e; ++i ) {
		if ( ss_state(i) > 0 ) {
			++num;
			if ( ss_state(i) == 1 && consHE(i) == secstruct(i) )
			 mismatch_frag_block += 0.66;
			if ( ss_state(i) == 2 && consHE(i) != secstruct(i) )
			 ++mismatch_frag_block;
		}
	}

//db  if consensus block is true, no more than one third of the consensus
//db  residues can differ from the consensus
//db  if consensus block is false, no more than half  of the consensus
//db  residues can be identical to the consensus
	if ( num > 0.5 ) {
		frac = mismatch_frag_block/num;
	} else {
		frac = 0.;
	}

	if ( frac > 0.5 ) {
		mismatch_frag_block = 0.0;
	} else {
		mismatch_frag_block = 1.0;
	}

	return mismatch_frag_block;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin scoreall_frag_block
///
/// @brief
///
/// @detailed
///
/// @param  size - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
scoreall_frag_block( int const size )
{
	using namespace block_for_eval;
	using namespace fragments;
	using namespace misc;
	using namespace param;

	FArray3D_float frag_block_score( MAX_RES()(), MAX_NEIGH()(), 3 );

	int size_bin = get_index_by_frag_size( size );

	for ( int i = 1; i <= total_residue; ++i ) {
		for ( int j = 1, je = align_depth( i, size_bin ); j <= je; ++j ) {
			float mismatch_frag_block = 0.0;
			float num = 0.0;
			frag_block_score( i, j, size_bin ) = 0.;
			for ( int k = 0; k < size; ++k ) {
				if ( ss_state(i+k) > 0 ) {
					++num;
					if ( ss_state(i+k) == 1 && consHE(i+k) == ss_type(i,j,k,size_bin) )
					 mismatch_frag_block += 0.66;
					if ( ss_state(i+k) == 2 && consHE(i+k) != ss_type(i,j,k,size_bin) )
					 mismatch_frag_block += 1.0;
				}
				if ( num > 0.1 ) frag_block_score(i,j,size_bin) = mismatch_frag_block/num;
			}

		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin score_frag_block
///
/// @brief
///db  this version does not use the heap. disadvantage is that cannot select
///db  nearly (but not totally) consistent blocks since no sorting is done.  not
///db currently used
///
/// @detailed
/// output
/// frag_block_pointer is the output array of pointers to frags
/// ordered by our scheme for ranking them using pointers
///
/// @param  size - [in/out]? - ize is 3 or 9 usually
/// @param  ires - [in/out]? - insertion position (first res in frag)
/// @param  frag_block_pointer - [in/out]? - frag list is a list of fragments at a
///        given insertion position
/// @param  block_frag_depth - [in/out]? - max_frag is the physical size of the array
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
score_frag_block(
	int const size,
	int & ires,
	FArray1Da_int frag_block_pointer,
	int & block_frag_depth
)
{
	using namespace block_for_eval;
	using namespace fragments;
	using namespace param;

	frag_block_pointer.dimension( MAX_NEIGH() );

	int size_bin = get_index_by_frag_size(size);
	int count = 0;
	for ( int j = 1, je = align_depth(ires,size_bin); j <= je; ++j ) {
		float mismatch_frag_block = 0.0;
		float num = 0.0;
		float block_score = 0.0;
		for ( int k = 0; k < size; ++k ) {
			if ( ss_state(ires+k) > 0 ) {
				++num;
				if ( ss_state(ires+k) == 1 && consHE(ires+k) == ss_type(ires,j,k,size_bin) )
					mismatch_frag_block += 0.66;
				if ( ss_state(ires+k) == 2 && consHE(ires+k) != ss_type(ires,j,k,size_bin) )
					mismatch_frag_block += 1.0;
			}
		}
		if ( num > 0.0 ) {
			block_score = mismatch_frag_block/num;
		}
		if ( block_score < 0.001 ) {
			++count;
			frag_block_pointer(count) = j;
		}

	}                     // end j (depth)

	if ( count < 25 ) {
		if ( count > 3 ) {
			block_frag_depth = count;
		} else {
			block_frag_depth = 3;
		}
	} else {
		block_frag_depth = 25;
	}
}

//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//  heap functions
//     heap is a list integers the value associated with each integer is
//     in the coheap (or covalue) array. Comparisons <=> are made between
//     the covalues.  The heap entries are thus sorted on the basis of
//     their covalues.  In typical usage, this is an index type sort
//     the heap values are the indicies, and the covalues the value.
//
//     heap is simply an array with two extra storage elments at the
//     front the first is the max dimension of the heap, the second is
//     the current number of entries (the third is the minimum value in
//     heap and the start of the heap). When dimensioning space for it be
//     sure to add 2 elements more than you think you need.
//
//     heap_init    set up an empty heap
//     heap_insert  insert a new value into the heap
//     heap_extract extract the lowset value (always located in heap(3))
//
//     heap_replace replace the lowest value
//        (equivalent to heap_extract; heap_insert  but faster)
//        If you call heap_insert with a full heap (ie last = maxsize) then
//        heap_replace gets called instead.

//     charlie strauss 1999
//------------------------------------------------------------------------------

////////////////////////////////////////////////////////////////////////////////
/// @begin heap_init
///
/// @brief
/// sets up an empty heap and stores the dimensioned size
///
/// @detailed
///
/// @param  heap - [in/out]? -
/// @param  coheap - [in/out]? -
/// @param  max_items - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
heap_init(
	FArray1Da_int heap,
	FArray1Da_float coheap,
	int max_items
)
{
	heap.dimension( SRange( -2, star ) );
	coheap.dimension( SRange( 0, star ) );

	heap(-1) = 0;
	heap(-2) = max_items;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin heap_extract
///
/// @brief
/// modifes heap and last_val
/// return val and err.
///
/// @detailed
///
/// @param  heap - [in/out]? - convert to zero offset matrix
/// @param  coheap - [in/out]? -
/// @param  val - [in/out]? -
/// @param  coval - [in/out]? -
/// @param  err - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
heap_extract(
	FArray1Da_int heap, // convert to zero offset matrix
	FArray1Da_float coheap,
	int & val,
	float & coval,
	bool & err
)
{
	heap.dimension( SRange( -2, star ) );
	coheap.dimension( SRange( 0, star ) );

	err = true;
	if ( heap(-1) < 1 ) return;

	int temp_val = heap(0);
	float temp_coval = coheap(0);
	err = false;
	--heap(-1);

	if ( heap(-1) == 0 ) { // special case for single value in heap
		val = temp_val; // PB bugfix
		coval = temp_coval;
		return;
	}

	heap(0) = heap(heap(-1)); // move last value to front
	coheap(0) = coheap(heap(-1));

	heap_down(heap,coheap,1);
//  we use a temporary copy so that this can be done in place if need be
	val = temp_val;
	coval = temp_coval;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin heap_insert
///
/// @brief
/// modifes heap and last_dummy, inserts val, returns err
/// requires heap_max to be previously set via heap_init
///
/// @detailed
///
/// @param  heap - [in/out]? - convert to zero offset matrix
/// @param  coheap - [in/out]? -
/// @param  val - [in/out]? -
/// @param  coval - [in/out]? -
/// @param  err - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
heap_insert(
	FArray1Da_int heap, // convert to zero offset matrix
	FArray1Da_float coheap,
	int val,
	float coval,
	bool & err
)
{
	heap.dimension( SRange( -2, star ) );
	coheap.dimension( SRange( 0, star ) );

	if ( heap(-1) >= heap(-2) ) { // list is full, use replace instead
		err = true;
		if ( coheap(0) < coval ) heap_replace(heap,coheap,val,coval);
		return;
	}

	err = false;
	heap(heap(-1)) = val; // empty spot on end (zero offset)
	coheap(heap(-1)) = coval;

	++heap(-1);
	heap_up(heap,coheap,heap(-1));

}

////////////////////////////////////////////////////////////////////////////////
/// @begin heap_replace
///
/// @brief
///
/// @detailed
///
/// @param  heap - [in/out]? - convert to zero offset matrix
/// @param  coheap - [in/out]? -
/// @param  val - [in/out]? -
/// @param  coval - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
heap_replace(
	FArray1Da_int heap, // convert to zero offset matrix
	FArray1Da_float coheap,
	int val,
	float coval
)
{
	heap.dimension( SRange( -2, star ) );
	coheap.dimension( SRange( 0, star ) );

// modifes heap

	bool err;
	err = false;

	heap(0) = val; // overwrite the lowest element
	coheap(0) = coval;
	heap_down(heap,coheap,1);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin heap_down
///
/// @brief
///
/// @detailed
///
/// @param  heap - [in/out]? - convert to zero offset matrix
/// @param  coheap - [in/out]? -
/// @param  index_in - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
heap_down(
	FArray1Da_int heap, // convert to zero offset matrix
	FArray1Da_float coheap,
	int index_in
)
{
	heap.dimension( SRange( -2, star ) );
	coheap.dimension( SRange( 0, star ) );
	float coiv,cocv,cocv2;
	int indx,child,iv,cv,cv2,last;
	indx = index_in-1; // convert to zero offset matrix
	last = heap(-1)-1; // convert to zero offset matrix

	if ( last <= 0 ) return; // empty or single element
	if ( indx > last ) return; // dumbass

	iv = heap(indx); // the inserted value
	coiv = coheap(indx);

	while ( indx < last ) {
		child = 2*indx+1;

		if ( child > last ) goto L20; // loop escape

		cv  = heap(child);
		cocv = coheap(child);

		if ( child < last ) {
			cv2 = heap (child+1);
			cocv2 = coheap(child+1);

			if ( cocv2 < cocv ) {
				cv = cv2;
				cocv = cocv2;

				++child;
			}
		}

		if ( coiv <= cocv ) goto L20; // loop escape
		coheap(indx) = cocv;
		heap(indx) = cv;
		indx = child;
	}

L20:; // loop escape
	heap(indx) = iv;
	coheap(indx) = coiv;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin heap_up
///
/// @brief
///
/// @detailed
///
/// @param  heap - [in/out]? - convert to zero offset matrix
/// @param  coheap - [in/out]? -
/// @param  index_in - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
heap_up(
	FArray1Da_int heap, // convert to zero offset matrix
	FArray1Da_float coheap,
	int & index_in
)
{
	heap.dimension( SRange( -2, star ) );
	coheap.dimension( SRange( 0, star ) );
	float covalue,copv;

	int indx,parent,value,pv;
	indx = index_in-1; // convert to zero offset matrix


	value = heap(indx);
	covalue = coheap(indx);

	while ( indx != 0 ) {
		parent = static_cast< int >((indx-1)/2);
		pv = heap(parent);
		copv = coheap(parent);
		if ( copv < covalue ) goto L20; // loop escape
		coheap(indx) = copv;
		heap(indx) = pv;
		indx = parent;
	}

L20:; // loop escape
	coheap(indx) = covalue;
	heap(indx) = value;
}
