// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 8387 $
//  $Date: 2006-05-16 14:08:33 -0700 (Tue, 16 May 2006) $
//  $Author: johnk $


// Rosetta Headers
#include "symmetric_design.h"
#include "aaproperties_pack.h"
#include "param.h"
#include "pose.h"
#include "refold.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/FArray3D.hh>

#include <vector>

// Using
using namespace param;

namespace design_sym {
	// num_clones: number of positions who follow this residue
	int num_clones = { 0 };
	// clone_follows: 0 for an independent position, otherwise resnum to follow
	FArray1D_int clone_follows( MAX_RES(), 0 );
	// clone_list: list of positions who follow this residue
	FArray2D_int clone_list( MAX_RES(), MAX_CLONES(), 0 );
	// clone_Mgl: translation/rotation to bring each clone to its base residue
	FArray3D_float clone_Mgl( 4, 4, MAX_RES() );
}



//////////////////////////////////////////////////////////////////////////////
/// @begin setup_symmetric_packing
///
/// @brief
///   Calculate the transform required to get from one backbone to another
///
/// @detailed
///   Intended for transferring a rotamer from one backbone position to
///   a symmetry-related position. For this reason, backbone geometry are
///   assumed to be identical.
///
///   Works by first translating to match CA's, then finding rotation which
///   matches C and N positions. Again, won't work if backbone geometries
///   differ!
///
/// @param  starting_full_coord - [in] - coors (a slice of xyz_inout array)
/// @param  arrival_full_coord - [in] - coors (a slice of xyz_inout array)
/// @param  trans_out - [out] - translation vector
/// @param  rot_out - [out] - rotation matrix
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///  jk
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
setup_symmetric_packing( int const total_residue )
{

	using namespace design_sym;

	if ( num_clones > 0 ) {
		int const total_residue_indep=total_residue/(num_clones+1);
		int lockres=1;
		int locknum=1;
		for ( int seqpos = total_residue_indep+1; seqpos <= total_residue; ++seqpos ) {
			// This is a dependent position
			clone_follows(seqpos)=lockres;
			clone_list(lockres,locknum)=seqpos;
			++lockres;
			if ( lockres > total_residue_indep ) {
				lockres=1;
				++locknum;
			}
		}
	}

	return;
}



///////////////////////////////////////////////////////////////////////////////
void
setup_symmetric_packing(
	int const nres,
	pose_ns::Symmetry_info const & s
)
{
	using namespace design_sym;

	assert( nres <= param::MAX_RES()() );

	// num_clones
	num_clones = s.num_chi_clones();
	if ( num_clones > param::MAX_CLONES()() ) {
		param::MAX_CLONES() = num_clones + 1;
	}

	if ( num_clones == 0 ) return;

	// for error-checking
	clone_follows = -1;
	clone_list = 0;

	// update clone_follows and clone_list
	for ( int seqpos=1; seqpos<= nres; ++seqpos ) {
		if ( s.chi_independent( seqpos ) ) {
			std::vector< int > const & clones( s.chi_clones( seqpos ) );
			assert( int(clones.size()) == num_clones );
			//assert( int(bb_clones.size()) == s.num_bb_clones() );
			assert( clone_follows( seqpos ) == -1 );
			clone_follows( seqpos ) = 0; // seqpos is independent
			for ( int i=1; i<= num_clones; ++i ) {
				int const clone_pos( clones[i-1] );
				clone_list( seqpos, i ) = clone_pos;
				assert( clone_follows( clone_pos ) == -1 );
				clone_follows( clone_pos ) = seqpos;
			}
		} // seqpos is independent?
	} // seqpos

	//	std::cout << "num_clones= " << num_clones << std::endl;

	// sanity check
	for ( int i=1; i<= nres; ++i ) {
		assert( clone_follows(i) != -1 );
	}
}



//////////////////////////////////////////////////////////////////////////////
void
reset_symmetric_packing()
{
	design_sym::num_clones = 0;
	design_sym::clone_follows = 0;
}


//////////////////////////////////////////////////////////////////////////////
/// @begin get_sym_rotamer_transform
///
/// @brief
///   Calculate the transform required to get from one backbone to another
///
/// @detailed
///   Intended for transferring a rotamer from one backbone position to
///   a symmetry-related position. For this reason, backbone geometry are
///   assumed to be identical.
///
///   Works by first translating to match CA's, then finding rotation which
///   matches C and N positions. Again, won't work if backbone geometries
///   differ!
///
/// @param  starting_full_coord - [in] - coors (a slice of xyz_inout array)
/// @param  arrival_full_coord - [in] - coors (a slice of xyz_inout array)
/// @param  Mgl - [out] - translation/rotation matrix
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///  jk
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
get_sym_rotamer_transform(
	FArray2Da_float starting_full_coord,
	FArray2Da_float arrival_full_coord,
	FArray2Da_float Mgl
)
{
	using namespace param;

	starting_full_coord.dimension( 3, MAX_ATOM() );
	arrival_full_coord.dimension( 3, MAX_ATOM() );
	Mgl.dimension( 4, 4 );

	// Recall: N is 1, CA is 2, C is 3
	int const N(1);
	int const CA(2);
	int const C(3);
	get_GL_matrix(starting_full_coord(1,N),starting_full_coord(1,CA),starting_full_coord(1,C),
		arrival_full_coord(1,N),arrival_full_coord(1,CA),arrival_full_coord(1,C),Mgl);

	// Done computing transforms!

	// If debug mode, make sure backbones are indeed symmetric
	assert ( verify_sym_rotamer_transform(starting_full_coord, arrival_full_coord, Mgl) );

	return;
}


//////////////////////////////////////////////////////////////////////////////
/// @begin verify_sym_rotamer_transform
///
/// @brief
///   Verify that the backbones for which the transform was computed
///     are indeed symmetry-related
///
/// @detailed
///   Since the transformation was computed using the N, CA, and C,
///   this works by applying the transformation to the O's and checking
///   whether the positions match.
///
/// @param  starting_xyz - [in] - coors (a slice of xyz_inout array)
/// @param  arrival_xyz - [in] - coors (a slice of xyz_inout array)
/// @param  Mgl - [out] - translation/rotation matrix
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///  jk
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool
verify_sym_rotamer_transform(
	FArray2Da_float incoord,
	FArray2Da_float targetcoord,
	FArray2Da_float Mgl
)
{
	using namespace param;

	float const sq_pos_err_thres(0.05);

	incoord.dimension( 3, MAX_ATOM() );
	targetcoord.dimension( 3, MAX_ATOM() );
	Mgl.dimension( 4, 4 );

	// Apply GL matrix to get final_coord
	FArray1D_float final_coord( 3 );
	GL_rot(Mgl,incoord(1,4),final_coord(1));

	float sq_pos_err(0);
	for ( int j = 1; j <= 3; ++j ) {
		sq_pos_err += (final_coord(j) - targetcoord(j,4)) * (final_coord(j) - targetcoord(j,4));
	}

	if ( sq_pos_err > sq_pos_err_thres ) return false;
	return true;

}


//////////////////////////////////////////////////////////////////////////////
/// @begin transfer_sym_rotamer
///
/// @brief
///   Apply the translation and rotation passed in to the input coors,
///     to get the output coors
///
/// @detailed
///   Intended to be used in conjunction with get_sym_rotamer_transform.
///   Note: Translation is applied first.
///
/// @param  Mgl - [in] - translation/rotation matrix
/// @param  incoord - [in] - coors
/// @param  inactcoord - [in] - coors
/// @param  outcoord - [out] - output coors
/// @param  outactcoord - [out] - output coors
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///  jk
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
transfer_sym_rotamer(
  int aa,
	int aav,
	FArray2Da_float Mgl,
	FArray2Da_float incoord,
	FArray1Da_float inactcoord,
	FArray2Da_float outcoord,
	FArray1Da_float outactcoord
)
{
	using namespace aaproperties_pack;
	using namespace param;

	Mgl.dimension( 4, 4 );
	incoord.dimension( 3, MAX_ATOM() );
	inactcoord.dimension( 3 );
	outcoord.dimension( 3, MAX_ATOM() );
	outactcoord.dimension( 3 );

	for ( int j = 1; j <= natoms(aa,aav); ++j ) {
		// Apply GL matrix
		GL_rot(Mgl,incoord(1,j),outcoord(1,j));
	}
	GL_rot(Mgl,inactcoord(1),outactcoord(1));

	return;
}


//////////////////////////////////////////////////////////////////////////////
/// @begin identity_Mgl
///
/// @brief
///   Check whether this translation + rotation matrix is (close to) the
///     identity matrix
///
/// @detailed
///   Intended to be used to test whether we should even bother applying the matrix
///
/// @param  Mgl - [in] - translation/rotation matrix
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///  jk
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool
identity_Mgl(
	FArray2Da_float Mgl
)
{

	Mgl.dimension( 4, 4 );

	float const zero_tol(0.0001);

	for ( int i = 1; i <= 4; ++i ) {
		for ( int j = 1; j <= 4; ++j ) {
			if ( i != j ) {
				if ( std::abs(Mgl(i,j)) > zero_tol ) return false;
			}
		}
	}

	for ( int i = 1; i <= 4; ++i ) {
		if ( std::abs(Mgl(i,i) - 1.) > zero_tol ) return false;
	}

	return true;
}

