// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 14367 $
//  $Date: 2007-04-19 12:08:54 +0300 (Thu, 19 Apr 2007) $
//  $Author: yab $


// Rosetta Headers
#include "termini.h"
#include "aaproperties_pack.h"
#include "angles.h"
#include "etable.h"
#include "fullatom.h"
#include "fullatom_energies.h"
#include "misc.h"
#include "orient_rms.h"
#include "pack_fwd.h"
#include "param.h"
#include "param_aa.h"
#include "param_pack.h"
#include "param_torsion.h"
#include "read_aaproperties.h"
#include "rotate.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/formatted.io.hh>
#include <ObjexxFCL/DimensionExpressions.hh>

// Numeric Headers
#include <numeric/conversions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>

// C++ Headers
#include <cstdlib>
#include <iostream>
#include <sstream>
#include <string>


namespace termini_ns {

	using namespace param;

	FArray1D_bool is_N_terminus( MAX_RES(), false );
	FArray1D_bool is_C_terminus( MAX_RES(), false );

	bool use_N_terminus( false ); // do we want to use the termini?
	bool use_C_terminus( false ); // do we want to use the termini?
}

////////////////////////////////////////////////////////////////////////////////
/// @begin build_termini
/// @brief called from input_pdb to build termini
///
/// @detailed
///
/// @param[in]  fullatom
///
/// @global_read
///
/// @global_write none
///
///
/// @remarks
///
/// @references
///
/// @authors bqian
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
build_termini(
	bool fullatom
)
{

	using namespace misc;
	using namespace aaproperties_pack;
	using namespace termini_ns;
	using namespace param_aa;

//bq local variables
	int respos;
	int aan;
	int num_of_HN;
	int aav_respos;
	int HN_pos;
	float phi_first = 0.0;
	float psi_last  = 0.0;

	static FArray2D_float Nterm_nhcoord( 3, 3 );
	static FArray2D_float Cterm_cocoord( 3, 2 );

	is_N_terminus = false;
	is_C_terminus = false;

	int start, end;
	for ( int  i = 1; i <= total_domains; ++i ) {
		if ( i == 1 ) {
			start = 1;
			end = domain_end(i);
		} else {
			start = domain_end(i-1) + 1;
			end   = domain_end(i);
		}

	//bq the residue Nterm
		respos = start;
		aan = misc::res(respos);
		if ( use_N_terminus && ( is_protein(aan) || is_nonnatural(aan) ) ){
			is_N_terminus(start) = true;

//bq trying to find the pure Nterm aa variant, which is not combined with
//bq anything else
			int const n_var = nvar( aan );
			bool found_nterm = false;
			for ( int j = 1; j <= n_var; ++j ){
				if ( variant_type( aav_Nterm, aan, j ) ){
					found_nterm = true;
					for ( int k = 1; k <= number_aav_type; ++k ){
						if ( variant_type( k, aan, j ) && k != aav_Nterm && k!=aav_base )
							found_nterm = false; // this is nterm combined with something else
					}

					if ( found_nterm ){
						res_variant(respos) = j;
						break;
					}
				}
			}
			if ( !found_nterm ) {
				std::cout << "Problem in aav initialization: no nterm defined!" << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
			aav_respos = res_variant(respos);

			assert( variant_type( aav_Nterm, aan, aav_respos) );

			if ( fullatom ){
				build_N_terminus_Hs( full_coord, aan, respos, phi_first, Nterm_nhcoord );
				num_of_HN = 3;
				if ( aan == aa_pro ) num_of_HN = 2;
				HN_pos = HNpos(aan,aav_respos);
				for ( int nh = 1; nh <= num_of_HN; ++nh ){
					if ( nh > 1 ) make_space_in_coords( nh+HN_pos-1, respos );
					for ( int k = 1, l = full_coord.index(k, nh+HN_pos-1,respos); k <= 3; ++k, ++l ) {
						full_coord[ l ] = Nterm_nhcoord(k,nh);
					}
				}
			}
		}

//bq build the C-term
		respos = end;
		aan = misc::res(respos);
		if ( use_C_terminus && ( is_protein(aan) || is_nonnatural(aan) )){
			is_C_terminus(end) = true;

//bq trying to find the pure Nterm aa variant, which is not combined with
//bq anything else
			int const n_var = nvar( aan );
			bool found_cterm = false;
			for ( int j = 1; j <= n_var; ++j ){
				if ( variant_type( aav_Cterm, aan, j ) ){
					found_cterm = true;
					for ( int k = 1; k <= number_aav_type; ++k ){
						if ( variant_type( k, aan, j ) && k != aav_Cterm && k!=aav_base )
							found_cterm = false; // this is nterm combined with something else
					}

					if ( found_cterm ){
						res_variant(respos) = j;
						break;
					}
				}
			}
			if ( !found_cterm ) {
				std::cout << "Problem in aav initialization: no cterm defined!" << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
			aav_respos = res_variant(respos);

			assert( variant_type( aav_Cterm, aan, aav_respos) );

			if ( fullatom ) {
				build_C_terminus_Os( full_coord, respos, psi_last, Cterm_cocoord);
				int const O_pos  = LookupByName(aan, aav_respos, " O  ");
				int const OXT_pos= LookupByName(aan, aav_respos, " OXT");

				for ( int k = 1, l = full_coord.index(k, O_pos, respos); k <= 3; ++k, ++l )
					full_coord[ l ] = Cterm_cocoord(k,1);
				int const Epos_O_pos( 5 );
				for ( int k = 1, l = position.index( k, MAX_POS*(respos-1)+Epos_O_pos); k <= 3; ++k, ++l )
					position[ l ] =  Cterm_cocoord(k,1);
				for ( int k = 1, l = Eposition.index( k, Epos_O_pos, respos ); k <= 3; ++k, ++l )
					Eposition[ l ] = Cterm_cocoord(k,1);
	//bq the OXT position
				make_space_in_coords( OXT_pos, respos );
				for ( int k = 1, l = full_coord.index(k, OXT_pos, respos); k <= 3; ++k, ++l )
					full_coord[ l ] = Cterm_cocoord(k,2);
			}
		}// if use_C_terminus
	}// for i 1..total_domains
}

////////////////////////////////////////////////////////////////////////////////
/// @begin setup_termini
/// @brief can setup all the termini parameters when called
///
/// @detailed
///
/// @param[in]
///
/// @global_read total_residues
///
/// @global_write none
///
///
/// @remarks
///
/// @references
///
/// @authors bqian
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
setup_termini(
)
{

	using namespace misc;
	using namespace aaproperties_pack;
	using namespace termini_ns;
	using namespace param_aa;

//bq local variables
	int respos;
	int aan;

	is_N_terminus = false;
	is_C_terminus = false;

	int start, end;

	for ( int  i = 1; i <= total_domains; ++i ) {
		if ( i == 1 ) {
			start = 1;
			end = domain_end(i);
		} else {
			start = domain_end(i-1) + 1;
			end   = domain_end(i);
		}


	//bq the residue Nterm
		respos = start;
		aan = misc::res(respos);
		if ( use_N_terminus && ( is_protein(aan) || is_nonnatural(aan) ) ){
			is_N_terminus(start) = true;

	//bq trying to find the pure Nterm aa variant, which is not combined with
	//bq anything else
			int const n_var = nvar( aan );
			bool found_nterm = false;
			for ( int j = 1; j <= n_var; ++j ){
				if ( variant_type( aav_Nterm, aan, j ) ){
					found_nterm = true;
					for ( int k = 1; k <= number_aav_type; ++k ){
						if ( variant_type( k, aan, j ) && k != aav_Nterm && k!=aav_base )
							found_nterm = false; // this is nterm combined with something else
					}

					if ( found_nterm ){
						res_variant(respos) = j;
						break;
					}
				}
			}
			if ( !found_nterm ) {
				std::cout << "Problem in aav initialization: no nterm defined!" << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
		}

	//bq build the C-term
		respos = end;
		aan = misc::res(respos);
		if ( use_C_terminus && ( is_protein(aan) || is_nonnatural(aan) )){
			is_C_terminus(end) = true;

	//bq trying to find the pure Nterm aa variant, which is not combined with
	//bq anything else
			int const n_var = nvar( aan );
			bool found_cterm = false;
			for ( int j = 1; j <= n_var; ++j ){
				if ( variant_type( aav_Cterm, aan, j ) ){
					found_cterm = true;
					for ( int k = 1; k <= number_aav_type; ++k ){
						if ( variant_type( k, aan, j ) && k != aav_Cterm && k!=aav_base )
							found_cterm = false; // this is nterm combined with something else
					}

					if ( found_cterm ){
						res_variant(respos) = j;
						break;
					}
				}
			}
			if ( !found_cterm ) {
				std::cout << "Problem in aav initialization: no cterm defined!" << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}

		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin create_terminus_variants
/// @brief create N and C terminus aa variants
///
/// @detailed
///    create the new variants which contain NH3+ group and COO- group
///
/// @param[in]   aa - in - amino acid
///
/// @global_read
/// variables from "param_aa.h"
///     aa_ser, aa_thr, aa_tyr, aa_trp, aa_lys, aa_asp, aa_glu
///     aa_gln, aa_asn, aa_arg, aa_his  -  the id of amino acids whose side chain
///                                        water are attached to
/// variables from "aaproperties_pack.h"
///     nvar - the current number of variants
///
/// @global_write none
///
///
/// @remarks
///
/// @references
///
/// @authors bqian
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
create_termini_variants( int aa )
{
//bq create amino acid variants that has the N-terminus NH3+ group or C-term
//bq COO group

	using namespace aaproperties_pack;

	int starting_variant;

	int old_nvar = nvar(aa);

//bq  skip DNA/RNA
	if ( aa > 20 ) return;

//bq   add new variations to all existing variations (ev)
	for ( starting_variant = 1; starting_variant <= old_nvar;
	 ++starting_variant ) {

//bq   make variant with NH3 termini. The base aa should already have HN, so
//bq   just add two Hs. Remember to change the coordinate of the first HN!!
		add_N_terminus_to_variant( aa, starting_variant );

//bq   make variant with COO termini. The base aa should already have O, so
//bq   just add one O. Remember to change the coordinate of the first O!!
		add_C_terminus_to_variant( aa, starting_variant );
	}  // starting variants

}

////////////////////////////////////////////////////////////////////////////////
/// @begin add_N_terminus_to_variant
///
/// @brief adding H to N-term N
///
/// @detailed
///    adding H to current variant, build up the topology of
///    these atoms by Lys NH3+ template group .
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors bqian
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
add_N_terminus_to_variant(
	int aa,
	int starting_variant
)
{

	using namespace aaproperties_pack;
	using namespace etable;
	using namespace param;
	using namespace param_aa;
	using namespace termini_ns;

	if ( !use_N_terminus ) return;

	int const var = create_new_variant(aa,starting_variant,aav_Nterm);

//bq change the NH groups

//bq change fullatom_type of backbone N
	int const HApos = LookupByName(aa, var, " HA ");
	int const Npos  = LookupByName(aa, var, " N  ");
	int const CApos = LookupByName(aa, var, " CA ");

	fullatom_type( Npos,aa,var ) = 10; // lys N
	if ( aa == aa_gly ){
		int const HA2pos = LookupByName(aa, var, "2HA ");
		int const HA3pos = LookupByName(aa, var, "3HA ");
		atomic_charge(  Npos, aa, var ) =  -0.30;
		atomic_charge( CApos, aa, var ) =   0.13;
		atomic_charge(HA2pos, aa, var ) =   0.09;
		atomic_charge(HA3pos, aa, var ) =   0.09;
	}else if ( aa == aa_pro ){
		int const HD2pos = LookupByName(aa, var, "2HD ");
		int const HD3pos = LookupByName(aa, var, "3HD ");
		int const  CDpos = LookupByName(aa, var, " CD ");
		atomic_charge(  Npos, aa, var ) =  -0.07;
		atomic_charge( CApos, aa, var ) =   0.16;
		atomic_charge( HApos, aa, var ) =   0.09;
		atomic_charge( CDpos, aa, var ) =   0.16;
		atomic_charge(HD2pos, aa, var ) =   0.09;
		atomic_charge(HD3pos, aa, var ) =   0.09;
	}else{
		atomic_charge(  Npos, aa, var ) =  -0.30;
		atomic_charge( CApos, aa, var ) =   0.21;
		atomic_charge( HApos, aa, var ) =   0.10;
	}


//bq make 1HN
	if( aa != aa_pro ){
		int const HNposition = HNpos(aa, var); // LookupByName(aa, var, " H  ");
		int const pos = HNposition;
		atom_name( pos, aa, var ) = "1H  ";
		fullatom_type( pos,aa,var ) = 22; // polar H
		atomic_charge(pos, aa, var) =  0.33; // HNbb
	}


	for ( int i = 0; i < 2; ++i ) {
	// make room after the HN
		std::string name;
		if ( aa != aa_pro ){
			if( i == 0 ) name = "3H  ";
			if( i == 1 ) name = "2H  ";
		} else {
			if( i == 0 ) name = "2H  ";
			if( i == 1 ) name = "1H  ";
		}
		int HNposition;
		if( aa != aa_pro ) HNposition = HNpos(aa, var);
		else HNposition = LookupByName(aa, var, " CD ");

		int const pos = HNposition + 1;

		make_room_for_atom( aa, var, pos );
		natoms(aa,var)++;
		atom_name( pos, aa, var ) = name;
		if( aa == aa_pro && name == "1H  " ) HNpos(aa, var) = pos;

//bq intra residue bonding
		int n_Neighbors_N = nbonded_neighbors( Npos, aa, var ) + 1; // N
		nbonded_neighbors( Npos, aa, var) = n_Neighbors_N; // N
			bonded_neighbor(n_Neighbors_N, Npos, aa, var) = pos; // N--H

		atom_base(pos, aa, var) = Npos; //  H  -> N
		fullatom_type( pos,aa,var ) = 22; // polar H
		if( aa == aa_pro ) atomic_charge(pos, aa, var) =  0.24; // HNbb
		else atomic_charge(pos, aa, var) =  0.33;

//bq   chi angles needed for building  H
		chi_required(1,pos, aa,var) = false;
		chi_required(2,pos, aa,var) = false;
		chi_required(3,pos, aa,var) = false;
		chi_required(4,pos, aa,var) = false;

		int nPolarH = nH_polar( aa,var ) + 1;
		nH_polar( aa,var ) = nPolarH; // number of polar hydrogens

//bq
		Hpos_polar( nPolarH, aa, var ) = pos; // 3HZ

		int n_H_on_N = nhydrogens_on_atm( Npos, aa, var ) + 1;
		nhydrogens_on_atm( Npos, aa, var ) = n_H_on_N; // N
		hydrogens_on_atm( n_H_on_N, Npos, aa, var ) = pos; //  H

// end of adding H
	}
//bq   build the termini
}


////////////////////////////////////////////////////////////////////////////////
/// @begin add_C_terminus_to_variant
///
/// @brief adding O to C-term CO
///
/// @detailed
///    adding O to current variant, build up the topology of
///    these atoms by Glu COO- template group .
///
/// @remarks
///
/// @references
///
/// @authors bqian
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
add_C_terminus_to_variant(
	int aa,
	int starting_variant
)
{

	using namespace aaproperties_pack;
	using namespace etable;
	using namespace param;
	using namespace param_aa;
	using namespace termini_ns;

	if ( !use_C_terminus ) return;

//bq create new variant
	int const var = create_new_variant(aa,starting_variant,aav_Cterm);

//bq change the CO groups
//bq change fullatom_type of backbone C
	int const Cpos = LookupByName(aa, var, " C  ");

	fullatom_type( Cpos,aa,var ) = 2; // Glu Carboxyl C
	atomic_charge( Cpos, aa, var ) =  0.34;

//bq make 1O
	int const Oposition = LookupByName(aa, var, " O  ");
	int pos = Oposition;
//	atom_name( pos, aa, var ) = " O  ";
	fullatom_type( pos,aa,var ) = 15; // Glu Carboxyl O
	atomic_charge(pos, aa, var) =  -0.67; // Carboxle O

// make room after the O
	std::string const name = " OXT";
//	pos = Oposition + 1;
	if( aa != aa_pro ) pos = HNpos(aa, var);
	else pos = LookupByName(aa, var, " CD ") + 1;
	assert( pos > 0 );

	make_room_for_atom( aa, var, pos );
	natoms(aa,var)++;
	nheavyatoms( aa,var )++;
	atom_name( pos, aa, var ) = name;

//bq intra residue bonding
	int n_Neighbors_C = nbonded_neighbors( Cpos, aa, var ) + 1; // C
	nbonded_neighbors( Cpos, aa, var) = n_Neighbors_C; // C
		bonded_neighbor(n_Neighbors_C, Cpos, aa, var) = pos; // C--O

	fullatom_type( pos,aa,var ) = 15; // Glu Carboxyl O
	atomic_charge(pos, aa, var) =  -0.67;

//bq  chi angles needed for building O
	chi_required(1,pos, aa,var) = false;
	chi_required(2,pos, aa,var) = false;
	chi_required(3,pos, aa,var) = false;
	chi_required(4,pos, aa,var) = false;

	int const nAcceptor = nacceptors(aa,var) + 1;
	nacceptors( aa,var ) = nAcceptor; // number of Hbond acceptors
	accpt_pos( nAcceptor,aa, var) =  pos; //  2O
	atom_base(pos, aa, var) = Cpos; //  O  -> C
//bq
// end of adding O
}

////////////////////////////////////////////////////////////////////////////////
/// @begin build_N_terminus_Hs
///
/// @brief
///bq build the N terminus hydrogens at resnum,
///bq based on the Nterm template provided by
///bq Brian and Doug
///
/// @detailed
///
/// @param  atm1 - [in/out]? -
/// @param  atm2 - [in/out]? -
/// @param  atm_out - [in/out]? -
/// @param  dis - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors bqian
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
build_N_terminus_Hs(
	FArray3DB_float const & xyzposition, // see comment below
	int const aan, // amino acid type
	int const resnum,
	float const phi_first,
	FArray2DB_float & Nterm_nhcoord
)
{

//bk  xyzposition is an array which specifies the positions of each atom
//bk  in the protein.
//bk  xyzposition(X,,) => xyz coordinates
//bk  xyzposition(,X,) => atom # within amino acid
//bk    atom  N CA C O
//bk       #  1  2 3 4
//bk  xyzposition(,,X) => residue number

	using namespace aaproperties_pack;
	using namespace param;
	using namespace param_aa;
	using namespace misc;
	using namespace termini_ns;

	if ( !is_N_terminus(resnum) ){
		std::cout << "ERROR:: Trying to build Nterm Hs on non-terminus residue "
		<< resnum << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	};


	FArray2D_double xyz( 3, 3 );
	FArray1D_double xn( 3 );
	FArray1D_double xca( 3 );
	FArray1D_double xc( 3 );
	FArray1D_double vec( 3 );
	FArray2D_double mat( 3, 3 );
	FArray1D_double off( 3 );

//bq currently not using phi_first to build Nterm Hs. But may use it later
	xn(1) = phi_first; // dummy to avoid compiler warnings. should get rid of it


// compute the matrix to rotate and align the
//  next (existing) residue onto last fragment stub

	for ( int n = 1; n <= 3; ++n ) {
		xn(n) = xyzposition(n,1,resnum);
		xca(n) = xyzposition(n,2,resnum);
		xc(n) = xyzposition(n,3,resnum);
	}

	if ( aan == aa_pro ) {
		for ( int n = 1; n <= 3; ++n ) { // need only three atoms
			xyz(n,1) = Nterm_icoor_pro(n,1); // N
			xyz(n,2) = Nterm_icoor_pro(n,2); // CA
			xyz(n,3) = Nterm_icoor_pro(n,3); // C
		}
	} else {
		for ( int n = 1; n <= 3; ++n ) { // need only three atoms
			xyz(n,1) = Nterm_icoor(n,1); // N
			xyz(n,2) = Nterm_icoor(n,2); // CA
			xyz(n,3) = Nterm_icoor(n,3); // C
		}
	}

	Dangles_align_transform(xyz(1,1),xyz(1,2),xyz(1,3),xn,xca,xc,mat);

// rotate the Nterm_icoor N so that we can find the offset
// vector that sets Nterm_icoor.N.rotate() = position_N
	rotate(mat,xyz(1,1),vec);
	for ( int j = 1; j <= 3; ++j ) {
		off(j) = xn(j) - vec(j);
	}

	FArray2D_float dest_Nterm( 3, Nterm_natoms ); // destination N_term

	if ( aan == aa_pro )
		transform(1,Nterm_natoms,mat,off,Nterm_icoor_pro(1,1),dest_Nterm(1,1));
	else
		transform(1,Nterm_natoms,mat,off,Nterm_icoor(1,1),dest_Nterm(1,1));

// only build the N-term hydrogens
	int numH( 3 );
	if ( aan == aa_pro ) numH = 2;
	for ( int n = 1; n <= numH; ++n ){
		for ( int m = 1; m <= 3; ++m ){
			Nterm_nhcoord( m, n ) = dest_Nterm( m, n+3 );
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin build_C_terminus_Os
///
/// @brief
///bq build the C terminus Oxygens,
///bq based on the Cterm template provided by
///bq Brian and Doug
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors bqian
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
build_C_terminus_Os(
	FArray3DB_float const & xyzposition, // see comment below
	int const resnum,
	float const psi_last,
	FArray2DB_float & Cterm_cocoord
){

//bk  xyzposition is an array which specifies the positions of each atom
//bk  in the protein.
//bk  xyzposition(X,,) => xyz coordinates
//bk  xyzposition(,X,) => atom # within amino acid
//bk    atom  N CA C O
//bk       #  1  2 3 4
//bk  xyzposition(,,X) => residue number

	using namespace aaproperties_pack;
	using namespace param;
	assert ( termini_ns::use_C_terminus );

	FArray2D_double xyz( 3, 3 );
	FArray1D_double xn( 3 );
	FArray1D_double xca( 3 );
	FArray1D_double xc( 3 );
	FArray1D_double vec( 3 );
	FArray2D_double mat( 3, 3 );
	FArray1D_double off( 3 );

// compute the matrix to rotate and align the
//  next (existing) residue onto last fragment stub

//bq currently not using psi_last to build the Cterm Os. May use it later
	xn(1) = psi_last; // dummy. should get rid of it

	for ( int n = 1; n <= 3; ++n ) {
		xn(n) = xyzposition(n,1,resnum);
		xca(n) = xyzposition(n,2,resnum);
		xc(n) = xyzposition(n,3,resnum);
	}

	for ( int n = 1; n <= 3; ++n ) { // need only three atoms
		xyz(n,1) = Cterm_icoor(n,1); // N
		xyz(n,2) = Cterm_icoor(n,2); // CA
		xyz(n,3) = Cterm_icoor(n,3); // C
	}

	Dangles_align_transform(xyz(1,1),xyz(1,2),xyz(1,3),xn,xca,xc,mat);

// rotate the Cterm_icoor C so that we can find the offset
// vector that sets Nterm_icoor.C.rotate() = position_C
	rotate(mat,xyz(1,3),vec);
	for ( int j = 1; j <= 3; ++j ) {
		off(j) = xc(j) - vec(j);
	}

	FArray2D_float dest_Cterm( 3, Cterm_natoms ); // destination C_term

	transform(1,Cterm_natoms,mat,off,Cterm_icoor(1,1),dest_Cterm(1,1));


// only build the C-term oxgen
	for ( int n = 1; n <= 2; ++n ){
		for ( int m = 1; m <= 3; ++m ){
			Cterm_cocoord( m, n ) = dest_Cterm( m, n+3 );
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin make_space_in_coords
///
/// @brief
///bq build the C terminus Oxygens,
///bq based on the Cterm template provided by
///bq Brian and Doug
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors bqian
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
make_space_in_coords(
	int const & atompos,
	int const & respos
)
{

	using namespace misc;

	for ( int pos = param::MAX_ATOM()-1; pos >= atompos; --pos ){
		for ( int k = 1, tar = full_coord.index(k, pos+1, respos), ori = full_coord.index(k, pos, respos); k <= 3; ++k, ++tar, ++ori )
			full_coord[ tar ] = full_coord[ ori ];
	}

}

