// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   numeric/xyz.functions.test.hh
/// @brief  test suite for numeric::xyz_functions
/// @author Stuart G. Mentzer (Stuart_Mentzer@objexx.com)
/// @author Kevin P. Hinshaw (KevinHinshaw@gmail.com)


// Test headers
#include <numeric/xyz.functions.test.hh>

// Package headers
#include <numeric/xyz.functions.hh>
#include <numeric/xyzVector.hh>
#include <numeric/xyzVector.io.hh>
#include <numeric/xyzMatrix.hh>
#include <numeric/xyzMatrix.io.hh>
#include <numeric/constants.hh>


namespace test {
namespace numeric {


// --------------- Fixtures --------------- //

// --- define a test fixture (some initial state that several tests share)
TEST_FIXTURE_BEGIN( fixture_xyz_functions )

// Shared initialization goes here.  (invoked when TEST_FIXTURE_USE is called)
TEST_FIXTURE_SETUP_BEGIN( fixture_xyz_functions )
{
	delta_percent = 0.0001;
}
TEST_FIXTURE_SETUP_END

// Shared finalization goes here.  (invoked when a test case that uses the fixture ends)
TEST_FIXTURE_TEARDOWN_BEGIN( fixture_xyz_functions )
{
}
TEST_FIXTURE_TEARDOWN_END

// Shared data elements go here.  (They will be declared public.)

double delta_percent; // percentage difference for floating-point comparisons in TEST_CHECK_CLOSE

TEST_FIXTURE_END  // fixture_xyz_functions


// --------------- Test Suites --------------- //

// --- set up the top-level test suite
TEST_SUITE_BEGIN( Tests_xyz_functions )
	TEST_SUITE_USES_CASE( test_xyz_functions_Eigenvalues );
	TEST_SUITE_USES_CASE( test_xyz_functions_Operations );
	TEST_SUITE_USES_CASE( test_xyz_functions_Rotation );
	TEST_SUITE_USES_CASE( test_xyz_functions_AngleOperations );
TEST_SUITE_END

	
// --------------- Test Cases --------------- //

/// @brief operations tests
TEST_CASE_BEGIN( test_xyz_functions_Operations )
{
	using ::numeric::xyzMatrix_float;
	using ::numeric::xyzVector_float;

	// compute an outer product
	xyzVector_float v( 1.0, 2.0, 3.0 );
	xyzVector_float w( 2.0, 1.0, 5.0 );
	xyzMatrix_float m = outer_product( v, w );
	TEST_CHECK_EQUAL( m, xyzMatrix_float::rows( 2.0, 1.0, 5.0, 4.0, 2.0, 10.0, 6.0, 3.0, 15.0 ) ); // equality here

	// matrix/vector product
	xyzVector_float u1 = m * v;
	TEST_CHECK_EQUAL( u1, xyzVector_float( 19.0, 38.0, 57.0 ) ); // equality here

	// matrix/vector in-place product
	xyzVector_float u2 = inplace_product( m, v );
	TEST_CHECK_EQUAL( u2, xyzVector_float( 19.0, 38.0, 57.0 ) ); // equality here
}
TEST_CASE_END


/// @brief test eigenvalue/eigenvector computations
TEST_CASE_BEGIN( test_xyz_functions_Eigenvalues )
{
	using ::numeric::xyzMatrix_double;
	using ::numeric::xyzVector_double;
	TEST_FIXTURE_USE( fixture_xyz_functions, fix );

	// build a symmetric matrix from three vectors
	xyzVector_double v1( 1.8, 2.0, 4.0 );
	xyzVector_double v2( 2.0, 5.0, 7.0 );
	xyzVector_double v3( 4.0, 7.0, 9.0 );
	xyzMatrix_double m1 = xyzMatrix_double::rows( v1, v2, v3 );

	// compute eigenvalues
	xyzVector_double ev1 = eigenvalue_jacobi( m1, 0.0 );
//	std::cout << "m1:\n" << m1 << std::endl;
//	std::cout << "eigenvalue_Jacobi returned: \n" << ev1 << std::endl;
	TEST_CHECK_CLOSE( ev1.x(),  0.9084668081160422, fix.delta_percent );
	TEST_CHECK_CLOSE( ev1.y(), -0.7863597288923477, fix.delta_percent );
	TEST_CHECK_CLOSE( ev1.z(), 15.67789292077631  , fix.delta_percent );

	// compute eigenvectors
	xyzMatrix_double ew1;
	ev1 = eigenvector_jacobi( m1, 0.0, ew1 );
//	std::cout << "Eigenvalues from eigenvector_Jacobi:\n" << ev1 << std::endl;
	TEST_CHECK_CLOSE( ev1.x(),  0.9084668081160422, fix.delta_percent );
	TEST_CHECK_CLOSE( ev1.y(), -0.7863597288923477, fix.delta_percent );
	TEST_CHECK_CLOSE( ev1.z(), 15.67789292077631  , fix.delta_percent );
//	std::cout << "Eigenvectors:\n" << ew1 << std::endl;
	TEST_CHECK_CLOSE( ew1.xx(),  0.7968224391550893, fix.delta_percent );
	TEST_CHECK_CLOSE( ew1.xy(),  0.5228228506830058, fix.delta_percent );
	TEST_CHECK_CLOSE( ew1.xz(),  0.3028700501248519, fix.delta_percent );
	TEST_CHECK_CLOSE( ew1.yx(), -0.5924536292558642, fix.delta_percent );
	TEST_CHECK_CLOSE( ew1.yy(),  0.5776511119658608, fix.delta_percent );
	TEST_CHECK_CLOSE( ew1.yz(),  0.5615317355467631, fix.delta_percent );
	TEST_CHECK_CLOSE( ew1.zx(),  0.1186284014917577, fix.delta_percent );
	TEST_CHECK_CLOSE( ew1.zy(), -0.6268775475707361, fix.delta_percent );
	TEST_CHECK_CLOSE( ew1.zz(),  0.7700336633623303, fix.delta_percent );

	// compute product of eigenvectors and a diagonal matrix of eigenvalues
	xyzMatrix_double diag = xyzMatrix_double::diag( ev1.x(), ev1.y(), ev1.z() );
	xyzMatrix_double ew1_diag = ew1 * diag;
//	std::cout << "Eigenvectors * diagonal:\n" << ew1_diag  << std::endl;
	TEST_CHECK_CLOSE( ew1_diag.xx(),  0.7238867379344632, fix.delta_percent );
	TEST_CHECK_CLOSE( ew1_diag.xy(), -0.4111268351218129, fix.delta_percent );
	TEST_CHECK_CLOSE( ew1_diag.xz(),  4.748364214767583 , fix.delta_percent );
	TEST_CHECK_CLOSE( ew1_diag.yx(), -0.5382244575268400, fix.delta_percent );
	TEST_CHECK_CLOSE( ew1_diag.yy(), -0.4542415717998375, fix.delta_percent );
	TEST_CHECK_CLOSE( ew1_diag.yz(),  8.803634421519833 , fix.delta_percent );
	TEST_CHECK_CLOSE( ew1_diag.zx(),  0.1077699652551254, fix.delta_percent );
	TEST_CHECK_CLOSE( ew1_diag.zy(),  0.4929512583564238, fix.delta_percent );
	TEST_CHECK_CLOSE( ew1_diag.zz(),  12.07250531958773 , fix.delta_percent );

	// compute product of original matrix and eigenvectors
	xyzMatrix_double m1_ew1 = m1 * ew1;
//	std::cout << "m1 * Eigenvectors:\n" << m1_ew1 << std::endl;
	TEST_CHECK_CLOSE( m1_ew1.xx(),  0.7238867379344631, fix.delta_percent );
	TEST_CHECK_CLOSE( m1_ew1.xy(), -0.4111268351218125, fix.delta_percent );
	TEST_CHECK_CLOSE( m1_ew1.xz(),  4.748364214767580 , fix.delta_percent );
	TEST_CHECK_CLOSE( m1_ew1.yx(), -0.5382244575268387, fix.delta_percent );
	TEST_CHECK_CLOSE( m1_ew1.yy(), -0.4542415717998374, fix.delta_percent );
	TEST_CHECK_CLOSE( m1_ew1.yz(),  8.803634421519831 , fix.delta_percent );
	TEST_CHECK_CLOSE( m1_ew1.zx(),  0.1077699652551269, fix.delta_percent );
	TEST_CHECK_CLOSE( m1_ew1.zy(),  0.4929512583564248, fix.delta_percent );
	TEST_CHECK_CLOSE( m1_ew1.zz(), 12.07250531958772  , fix.delta_percent );

	// the two products above should match
	TEST_CHECK_CLOSE( m1_ew1.xx(), ew1_diag.xx(), fix.delta_percent );
	TEST_CHECK_CLOSE( m1_ew1.xy(), ew1_diag.xy(), fix.delta_percent );
	TEST_CHECK_CLOSE( m1_ew1.xz(), ew1_diag.xz(), fix.delta_percent );
	TEST_CHECK_CLOSE( m1_ew1.yx(), ew1_diag.yx(), fix.delta_percent );
	TEST_CHECK_CLOSE( m1_ew1.yy(), ew1_diag.yy(), fix.delta_percent );
	TEST_CHECK_CLOSE( m1_ew1.yz(), ew1_diag.yz(), fix.delta_percent );
	TEST_CHECK_CLOSE( m1_ew1.zx(), ew1_diag.zx(), fix.delta_percent );
	TEST_CHECK_CLOSE( m1_ew1.zy(), ew1_diag.zy(), fix.delta_percent );
	TEST_CHECK_CLOSE( m1_ew1.zz(), ew1_diag.zz(), fix.delta_percent );
}
TEST_CASE_END


/// @brief test matrix rotation
TEST_CASE_BEGIN( test_xyz_functions_Rotation )
{
	using namespace ::numeric::constants::d;
	using ::numeric::xyzMatrix_double;
	using ::numeric::xyzVector_double;
	TEST_FIXTURE_USE( fixture_xyz_functions, fix );

	// compute a rotation matrix, given an axis/angle pair
	xyzVector_double axis1( -4.0, 2.0, -1.0 );
	double theta1 = 45.0 * degrees_to_radians;
	xyzMatrix_double rm = rotation_matrix( axis1, theta1 );
	TEST_CHECK_CLOSE( rm.xx(),  0.9302635193301304 , fix.delta_percent );
	TEST_CHECK_CLOSE( rm.xy(),  0.04272498089030048, fix.delta_percent );
	TEST_CHECK_CLOSE( rm.xz(),  0.3643958844600796 , fix.delta_percent );
	TEST_CHECK_CLOSE( rm.yx(), -0.2658817190338834 , fix.delta_percent );
	TEST_CHECK_CLOSE( rm.yy(),  0.7628959657224432 , fix.delta_percent );
	TEST_CHECK_CLOSE( rm.yz(),  0.5893188075804199 , fix.delta_percent );
	TEST_CHECK_CLOSE( rm.zx(), -0.2528175153882882 , fix.delta_percent );
	TEST_CHECK_CLOSE( rm.zy(), -0.6451079921163156 , fix.delta_percent );
	TEST_CHECK_CLOSE( rm.zz(),  0.7210540773205214 , fix.delta_percent );

	// given the rotation matrix, retrieve the axis/angle rotation pair
	double theta2 = 0.0;
	xyzVector_double axis2 = rotation_axis( rm, theta2 );  // modifies theta2

	// the rotation angle should match the original
	TEST_CHECK_CLOSE( theta1, theta2, fix.delta_percent );

	// the rotation axis should match a normalized version of the original
	xyzVector_double axis1_norm = axis1.normalized();
	TEST_CHECK_CLOSE( axis1_norm.x(), axis2.x(), fix.delta_percent );
	TEST_CHECK_CLOSE( axis1_norm.y(), axis2.y(), fix.delta_percent );
	TEST_CHECK_CLOSE( axis1_norm.z(), axis2.z(), fix.delta_percent );
}
TEST_CASE_END


/// @brief angle tests
TEST_CASE_BEGIN(test_xyz_functions_AngleOperations)
{
	using ::numeric::xyzVector_float;
	using ::numeric::dihedral;
	TEST_FIXTURE_USE( fixture_xyz_functions, fix );

	// compute a dihedral angle
	{
	xyzVector_float p1( -16.577999, 5.2760000, -10.507000 );
	xyzVector_float p2( -16.726999, 6.5549998, -10.734000 );
	xyzVector_float p3( -17.601999, 7.3889999, -9.8529997 );
	xyzVector_float p4( -16.798000, 7.8550000, -8.6330004 );
	TEST_CHECK_CLOSE( dihedral( p1,  p2,  p3,  p4 ), 85.0868f, fix.delta_percent );
	}

	// compute another dihedral angle
	{
	xyzVector_float p1( -2.5362606, 15.318624, 30.642799 );
	xyzVector_float p2( -2.6063213, 15.113583, 32.148636 );
	xyzVector_float p3( -2.8367538, 16.429384, 32.877182 );
	xyzVector_float p4( -2.9080868, 16.243940, 34.351967 );

	// test normal case
	TEST_CHECK_CLOSE( dihedral( p1,  p2,  p3,  p4 ), 179.976f, fix.delta_percent );

	// test degenerate cases (angle should be zero)
	TEST_CHECK_CLOSE( dihedral( p1,  p1,  p3,  p4 ), 0.0f, fix.delta_percent );
	TEST_CHECK_CLOSE( dihedral( p1,  p2,  p3,  p2 ), 0.0f, fix.delta_percent );
	}
}
TEST_CASE_END


} // namespace numeric
} // namespace test
