// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   utility/BitVector.test.cc
/// @brief  BitVector.test: test suite for utility::BitVector
/// @author Stuart G. Mentzer (Stuart_Mentzer@objexx.com)


// Package headers
#include <utility/BitVector.hh>

// Test headers
#include <utility/BitVector.test.hh>


namespace test {
namespace utility {


enum Bit { zero, one, two, three, four, nine=9, ten };

inline
::utility::BitVector< Bit >
operator |( Bit const & a, Bit const & b )
{
	return ::utility::BitVector< Bit >( a, b );
}


namespace b {

class Bit
{
public:

	inline
	explicit
	Bit( int ii ) :
		i( ii )
	{}

	friend
	inline
	bool
	operator <( Bit const & a, Bit const & b )
	{
		return ( a.i < b.i );
	}

	friend
	inline
	::utility::BitVector< Bit >
	operator |( Bit const & a, Bit const & b )
	{
		return ::utility::BitVector< Bit >( a, b );
	}

	inline
	operator int() const
	{
		return i;
	}

	int i;
};

} // namespace b


// Test suite
TEST_SUITE_BEGIN(BitVector_Tests)
	TEST_SUITE_USES_CASE(test_BitVector_constructor)
	TEST_SUITE_USES_CASE(test_BitVector_assignment)
	TEST_SUITE_USES_CASE(test_BitVector_method)
TEST_SUITE_END


// Test cases


/// @brief Constructor Test
TEST_CASE_BEGIN(test_BitVector_constructor)
{

	{ // enum Bit: Copy construction
		::utility::BitVector< Bit > s( one | nine |= ten );
		TEST_CHECK( s[ one ] );
		TEST_CHECK( ! s[ two ] );
		TEST_CHECK( s[ nine ] );
		TEST_CHECK( s[ ten ] );
		TEST_CHECK( s.size() == 11 );
		TEST_CHECK( ! s.empty() );
	}

	{ // class Bit Bit: Copy construction
		b::Bit one(1), two(2), nine(9), ten(10);
		::utility::BitVector< b::Bit > s( one | nine |= ten );
		TEST_CHECK( s[ one ] );
		TEST_CHECK( ! s[ two ] );
		TEST_CHECK( s[ nine ] );
		TEST_CHECK( s[ ten ] );
	}

	{ // enum Bit: Multiple bit construction
		::utility::BitVector< Bit > s( one, nine, ten );
		TEST_CHECK( s[ one ] );
		TEST_CHECK( ! s[ two ] );
		TEST_CHECK( s[ nine ] );
		TEST_CHECK( s[ ten ] );
	}

}
TEST_CASE_END


/// @brief Assignment Test
TEST_CASE_BEGIN(test_BitVector_assignment)
{

	{ // |= BitVector
		::utility::BitVector< Bit > s( one | nine |= ten );
		::utility::BitVector< Bit > const t( two );
		TEST_CHECK( s[ one ] );
		TEST_CHECK( ! s[ two ] );
		TEST_CHECK( s[ nine ] );
		TEST_CHECK( s[ ten ] );
		TEST_CHECK( t[ two ] );
		s |= t;
		TEST_CHECK( s[ two ] );
	}

	{ // += BitVector
		::utility::BitVector< Bit > s( one | nine |= ten );
		::utility::BitVector< Bit > const t( two );
		TEST_CHECK( s[ one ] );
		TEST_CHECK( ! s[ two ] );
		TEST_CHECK( s[ nine ] );
		TEST_CHECK( s[ ten ] );
		TEST_CHECK( t[ two ] );
		s += t;
		TEST_CHECK( s[ two ] );
	}

	{ // -= BitVector
		::utility::BitVector< Bit > s( one | nine |= ten );
		::utility::BitVector< Bit > const t( nine );
		TEST_CHECK( s[ one ] );
		TEST_CHECK( ! s[ two ] );
		TEST_CHECK( s[ nine ] );
		TEST_CHECK( s[ ten ] );
		TEST_CHECK( t[ nine ] );
		s -= t;
		TEST_CHECK( ! s[ nine ] );
	}

	{ // |=, -=, += Bit
		::utility::BitVector< Bit > s( one | nine |= ten );
		TEST_CHECK( s[ one ] );
		TEST_CHECK( ! s[ two ] );
		TEST_CHECK( s[ nine ] );
		TEST_CHECK( s[ ten ] );
		s |= two;
		TEST_CHECK( s[ two ] );
		s -= nine;
		TEST_CHECK( ! s[ nine ] );
		s += nine;
		TEST_CHECK( s[ nine ] );
	}
}
TEST_CASE_END


/// @brief Method Test
TEST_CASE_BEGIN(test_BitVector_method)
{

	{ // BitVector | BitVector
		::utility::BitVector< Bit > const s( one | nine );
		::utility::BitVector< Bit > const t( two | ten );
		::utility::BitVector< Bit > const u( s | t );
		TEST_CHECK( u[ one ] );
		TEST_CHECK( u[ two ] );
		TEST_CHECK( u[ nine ] );
		TEST_CHECK( u[ ten ] );
		TEST_CHECK( ! u[ three ] );
	}

	{ // BitVector + BitVector
		::utility::BitVector< Bit > const s( one | nine );
		::utility::BitVector< Bit > const t( two | ten );
		::utility::BitVector< Bit > const u( s + t );
		TEST_CHECK( ! s[ ten ] );
		TEST_CHECK( u[ one ] );
		TEST_CHECK( u[ two ] );
		TEST_CHECK( u[ nine ] );
		TEST_CHECK( u[ ten ] );
		TEST_CHECK( ! u[ three ] );
	}

	{ // BitVector - BitVector
		::utility::BitVector< Bit > const s( one | nine );
		::utility::BitVector< Bit > const t( two | nine );
		::utility::BitVector< Bit > const u( s - t );
		TEST_CHECK( u[ one ] );
		TEST_CHECK( ! u[ two ] );
		TEST_CHECK( ! u[ nine ] );
		TEST_CHECK( ! u[ ten ] );
	}

	{ // Swap
		::utility::BitVector< Bit > s( one | nine );
		::utility::BitVector< Bit > t( two | nine );
		TEST_CHECK( s != t );
		TEST_CHECK( s[ one ] );
		TEST_CHECK( ! s[ two ] );
		TEST_CHECK( s[ nine ] );
		TEST_CHECK( t[ two ] );
		TEST_CHECK( ! t[ three ] );
		TEST_CHECK( t[ nine ] );
		swap( s, t );
		TEST_CHECK( t[ one ] );
		TEST_CHECK( ! t[ two ] );
		TEST_CHECK( t[ nine ] );
		TEST_CHECK( s[ two ] );
		TEST_CHECK( ! s[ three ] );
		TEST_CHECK( s[ nine ] );
	}

	{ // BitVector ==, != BitVector
		::utility::BitVector< Bit > s( one | nine |= ten );
		::utility::BitVector< Bit > t;
		TEST_CHECK( t.empty() );
		t = s;
		TEST_CHECK( s == t );
		t |= two;
		TEST_CHECK( s != t );
	}

}
TEST_CASE_END


} // namespace utility
} // namespace test
