// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 13616 $
//  $Date: 2007-03-18 08:39:36 +0200 (Sun, 18 Mar 2007) $
//  $Author: stuartm $


// Rosetta Headers
#include "tree.h"
#include "param.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/formatted.o.hh>

// C++ Headers
#include <algorithm>
#include <iostream>


//Utility Headers
#include <utility/basic_sys_util.hh>

////////////////////////////////////////////////////////////////////////////////
/// @begin num_edges
///
/// @brief
///
/// @detailed
///
/// @param  g - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
int
num_edges( FArray2Da_int g )
{
	g.dimension( 3, SRange( 0, star ) );

	return g(1,0);
}


////////////////////////////////////////////////////////////////////////////////
/// @begin add_edge
///
/// @brief
///
/// @detailed
///
/// @param  g - [in/out]?
/// @param  estart - [in/out]?
/// @param  estop - [in/out]?
/// @param  elabel - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
add_edge(
	FArray2Da_int g,
	int const estart,
	int const estop,
	int const elabel
)
{
	using namespace param;      // for max_fold_tree_edges

	g.dimension( 3, SRange( 0, star ) );

	int const n = g(1,0);

	if ( n >= max_fold_tree_edges ) { // debug
		std::cout << "too many edges?? " << n << std::endl;
		show_graph(g);
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	g(1,0) = n + 1; // increment the size

	g(1,n+1) = estart;
	g(2,n+1) = estop;
	g(3,n+1) = elabel;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin get_edge
///
/// @brief
///
/// @detailed
///
/// @param  g - [in/out]?
/// @param  estart - [in/out]?
/// @param  estop - [in/out]?
/// @param  elabel - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_edge(
	FArray2Da_int g,
	int const i,
	int & estart,
	int & estop,
	int & elabel
)
{
	g.dimension( 3, SRange( 0, star ) );

	if ( i > g(1,0) ) {    // debug
		std::cout << "g too small:" << SS( i ) << SS( g(1,0) ) << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	estart = g(1,i);
	estop = g(2,i);
	elabel = g(3,i);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_unordered_edge
///
/// @brief
/// returns with estart <= estop
///
/// @detailed
///
/// @param  g - [in/out]?
/// @param  estart - [in/out]?
/// @param  estop - [in/out]?
/// @param  elabel - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_unordered_edge(
	FArray2Da_int g,
	int const i,
	int & estart,
	int & estop,
	int & elabel
)
{
	g.dimension( 3, SRange( 0, star ) );

	estart = std::min( g(1,i), g(2,i) );
	estop = std::max( g(1,i), g(2,i) );
	elabel = g(3,i);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin delete_edge
///
/// @brief
///
/// @detailed
///
/// @param  g - [in/out]?
/// @param  edge_num - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
delete_edge(
	FArray2Da_int g,
	int const edge_num
)
{
	g.dimension( 3, SRange( 0, star ) );

	int n = g(1,0);
	g(1,0) = n - 1;

	for ( int i = edge_num; i < n; ++i ) {
		for ( int j = 1; j <= 3; ++j ) {
			g(j,i) = g(j,i+1);
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin show_graph
///
/// @brief
///
/// @detailed
///
/// @param  g - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
show_graph( FArray2Da_int g )
{
	g.dimension( 3, SRange( 0, star ) );

	int estart,estop,elabel;

	for ( int i = 1, ie = num_edges(g); i <= ie; ++i ) {
		get_edge(g,i,estart,estop,elabel);
		std::cout << "show_graph: " << ' ' << I( 5, i ) << ' ' << I( 5, estart ) <<
		 ' ' << I( 5, estop ) << ' ' << I( 5, elabel ) << std::endl;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin output_graph
///
/// @brief
///
/// @detailed
///
/// @param[in]   iunit - in
/// @param  g - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
output_graph(
	std::ostream & iunit, // input
	FArray2Da_int g
)
{
	g.dimension( 3, SRange( 0, star ) );

	int estart,estop,elabel;

	for ( int i = 1, ie = num_edges(g); i <= ie; ++i ) {
		get_edge(g,i,estart,estop,elabel);
		iunit << "show_graph: " << ' ' << I( 5, i ) << ' ' << I( 5, estart ) <<
		 ' ' << I( 5, estop ) << ' ' << I( 5, elabel ) << std::endl;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin update_edge_label
///
/// @brief
///
/// @detailed
///
/// @param  g - [in/out]?
/// @param  edge_num - [in/out]?
/// @param  label - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
update_edge_label(
	FArray2Da_int g,
	int const edge_num,
	int const label
)
{
	g.dimension( 3, SRange( 0, star ) );

	if ( edge_num > num_edges(g) ) {
		std::cout << "edge num is too big!" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	g(3,edge_num) = label;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin graph_has_unordered_edge
///
/// @brief
///
/// @detailed
///
/// @param  g - [in/out]?
/// @param  edge_start - [in/out]?
/// @param  edge_stop - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
graph_has_unordered_edge(
	FArray2Da_int g,
	int const edge_start,
	int const edge_stop
)
{
	g.dimension( 3, SRange( 0, star ) );

	int estart,estop,elabel;

	for ( int i = 1, ie = num_edges(g); i <= ie; ++i ) {
		get_edge(g,i,estart,estop,elabel);
		if ( ( estart == edge_start && estop == edge_stop ) ||
		 ( estart == edge_stop && estop == edge_start ) ) {
			return true;
		}
	}
	return false;
}

/////////////////////////////////////////////////////////////////////////////
/// @begin copy_tree
///
/// @brief
///
/// @detailed
///
/// @param[in]   h - in
/// @param  g - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// the caller is responsible for making sure g is big enough
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
///
void
copy_tree(
	FArray2Da_int h, // input
	FArray2Da_int g
)
{
	h.dimension( 3, SRange( 0, star ) );
	g.dimension( 3, SRange( 0, star ) );

	int estart,estop,elabel;

	g(1,0) = 0;

	for ( int i = 1, h_size = num_edges(h); i <= h_size; ++i ) {
		get_edge(h,i,estart,estop,elabel);
		add_edge(g,estart,estop,elabel);
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin reorder_tree
///
/// @brief
///
/// @detailed
/// return another version of g such that
///
/// if we build up a tree by adding the edges
/// in order, it will always be the case that
/// the start vertex of a new edge is already a
/// member of the tree (except for the very first edge)
///
/// it may be necessary to reorder some of the start/stop
/// vertices, so the return tree wont necessarily have
/// exactly the same edge set
/// debugging: have to allow for chain-segment edges that
/// have estart == estop
///
/// @param  g - [in/out]?
/// @param  start_vertex - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
reorder_tree(
	FArray2Da_int g,
	int const start_vertex
)
{
	using namespace param;         // for max_fold_tree_edges, MAX_RES()

	g.dimension( 3, SRange( 0, star ) );

	int estart,estop,elabel;

	FArray2D_int h( 3, DRange( 0, max_fold_tree_edges ) );
	FArray1D_bool linked( MAX_RES()(), false );

	h(1,0) = 0; // empty tree

	linked(start_vertex) = true;
	bool new_member = true;
	int g_size = num_edges(g);

	while ( new_member ) {
		new_member = false;
		for ( int i = 1; i <= g_size; ++i ) {
			get_edge(g,i,estart,estop,elabel);

			if ( linked(estart) && ! linked(estop) ) {

				add_edge(h,estart,estop,elabel);
				linked(estop) = true;
				new_member = true;

			} else if ( linked(estop) && ! linked(estart) ) {

				add_edge(h,estop,estart,elabel); // order of vertices is reversed!!
				linked(estart) = true;
				new_member = true;

			} else if ( linked(estart) && linked(estop) ) {
				if ( ! graph_has_unordered_edge(h,estart,estop) ) {
					if ( estart == estop ) {
						add_edge(h,estart,estop,elabel);
					} else {
						std::cout << "tree has cycles" << std::endl;
						show_graph(g);
						utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
					}
				}            // is the edge already in h??
			}               // are the vertices linked?
		}
	}

	copy_tree(h,g);
}


////////////////////////////////////////////////////////////////////////////////
/// @begin split_tree
///
/// @brief
///
/// @detailed
///
/// @param  g - [in/out]?
/// @param  edge_num - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
split_tree(
	FArray2Da_int g,
	int & edge_num // INPUT / OUTPUT !!!!!!!!!!!!!!
)
{
	using namespace param;         // for max_fold_tree_edges

	g.dimension( 3, SRange( 0, star ) );

	int estart,estop,elabel;
	int e_start,e_stop,e_label;
	FArray2D_int t1( 3, DRange( 0, max_fold_tree_edges ) );
	FArray2D_int t2( 3, DRange( 0, max_fold_tree_edges ) );

	if ( num_edges(g) > max_fold_tree_edges ) { // debug
		std::cout << " g too big" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	get_edge(g,edge_num,estart,estop,elabel);

	e_start = std::min(estart,estop);
	e_stop = std::max(estart,estop);
	e_label = elabel;

	copy_tree(g,t1);
	delete_edge(t1,edge_num);
	reorder_tree(t1,e_start);

	copy_tree(g,t2);
	delete_edge(t2,edge_num);
	reorder_tree(t2,e_stop);

// add the splitting edge and the t2-edges to t1
	add_edge(t1,e_start,e_stop,e_label);
	edge_num = num_edges(t1); // edge_num is output variable also!!!

	for ( int i = 1, ie = num_edges(t2); i <= ie; ++i ) {
		get_edge(t2,i,estart,estop,elabel);
		add_edge(t1,estart,estop,elabel);
	}

// error checking -----------------
	if ( num_edges(t1) != num_edges(g) ) {
		std::cout << "split_tree oops" <<
		 SS( num_edges(t1) ) << SS( num_edges(g) ) << std::endl;
		show_graph(t1);
		show_graph(g);
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	for ( int i = 1, ie = num_edges(g); i <= ie; ++i ) {
		get_edge(g,i,estart,estop,elabel);
		if ( ! graph_has_unordered_edge(t1,estart,estop) ) {
			std::cout << "split_tree oops2" << std::endl;
			show_graph(t1);
			show_graph(g);
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}

	copy_tree(t1,g);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin modify_edge
///
/// @brief
///
/// @detailed
///
/// @param  g - [in/out]?
/// @param[in]   edge_num - in
/// @param  estart - [in/out]?
/// @param  estop - [in/out]?
/// @param  elabel - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
modify_edge(
	FArray2Da_int g,
	int const edge_num, // input
	int const estart,
	int const estop,
	int const elabel
)
{
	g.dimension( 3, SRange( 0, star ) );

	g(1,edge_num) = estart;
	g(2,edge_num) = estop;
	g(3,edge_num) = elabel;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin find_shortest_path
///
/// @brief
/// the path goes from the start to the end
/// this function only works for trees, currently
///
/// @detailed
///
/// @param  g - [in/out]?
/// @param  start_vertex - [in/out]?
/// @param  end_vertex - [in/out]?
/// @param[out]   path - out
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
find_shortest_path(
	FArray2Da_int g,
	int const start_vertex,
	int const end_vertex,
	FArray2Da_int path // output
)
{
	using namespace param;         // MAX_RES

	g.dimension( 3, SRange( 0, star ) );
	path.dimension( 3, SRange( 0, star ) );

	int const unassigned = { -1 };
	int const uninitialized = { -2 };

	FArray1D_int end_distance( MAX_RES()(), uninitialized );
	int estart, estop, elabel, vertex, next_vertex, d1, d2, path_length;


	std::cout << "find_shortest_path:" <<
	 SS( start_vertex ) << SS( end_vertex ) << std::endl;

	for ( int i = 1, ie = num_edges(g); i <= ie; ++i ) {
		get_edge(g,i,estart,estop,elabel);
		end_distance(estart) = unassigned;
		end_distance(estop) = unassigned;
	}

	if ( end_distance(start_vertex) == uninitialized ||
	 end_distance(end_vertex) == uninitialized ) { // debug
		std::cout << "start or end vertex not in tree!!" <<
		 SS( start_vertex ) << SS( end_vertex ) << std::endl;
		show_graph(g);
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	end_distance(end_vertex) = 0;

	while ( end_distance(start_vertex) == unassigned ) {
		for ( int i = 1, ie = num_edges(g); i <= ie; ++i ) {
			get_edge(g,i,estart,estop,elabel);
			d1 = end_distance(estart);
			d2 = end_distance(estop);
			if ( d1 == unassigned && d2 != unassigned ) {
				end_distance(estart) = d2 + 1;
			} else if ( d1 != unassigned && d2 == unassigned ) {
				end_distance(estop) = d1 + 1;
			}
		}
	}

// reconstruct the path
	path_length = end_distance(start_vertex);
	std::cout << "path_length:" << SS( path_length ) << std::endl;
	vertex = start_vertex;

	path(1,0) = 0;

	for ( int i = path_length; i >= 1; --i ) {

		next_vertex = -1;
		for ( int j = 1, je = num_edges(g); j <= je; ++j ) {
			get_edge(g,j,estart,estop,elabel);
//			std::cout << SS( vertex ) << SS( i ) << SS( estart ) <<
//			 SS( end_distance(estart) ) << SS( estop ) <<
//			 SS( end_distance(estop) ) << SS( elabel ) << std::endl;

			if ( estart == vertex && end_distance(estop) == i-1 ) {
				next_vertex = estop;
				goto L10;
			} else if ( estop == vertex && end_distance(estart) == i-1 ) {
				next_vertex = estart;
				goto L10;
			}
		}
L10:

		if ( next_vertex == -1 ) { // debug
			std::cout << "problem in shortest path" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		add_edge(path,vertex,next_vertex,elabel);
		vertex = next_vertex;
	}

	std::cout << "shortest path:" << std::endl;
	show_graph(path);


	if ( num_edges(path) != path_length ) utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	get_edge(path,1,estart,estop,elabel);
	if ( estart != start_vertex ) utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	get_edge(path,path_length,estart,estop,elabel);
	if ( estop != end_vertex ) utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
}

//////////////////////////////////////////////////////////////////////////////
/// @begin find_shortest_path
///
/// @brief
// this doesnt check the edge labels or order
// since it uses graph_has_unordered_edge which doesnt
// care about labels.
///
/// @detailed
///
/// @param  t1 - [in/out]?
/// @param  t2 - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
//////////////////////////////////////////////////////////////////////////////
bool
same_tree(
	FArray2Da_int t1,
	FArray2Da_int t2
)
{
	t1.dimension( 3, SRange( 0, star ) );
	t2.dimension( 3, SRange( 0, star ) );

// local:
	int estart, estop, elabel;

	int const n1 = num_edges(t1);
	int const n2 = num_edges(t2);

	if ( n1 != n2 ) return false;

	for ( int i = 1; i <= n1; ++i ) {
		get_edge(t1,i,estart,estop,elabel);
		if ( ! graph_has_unordered_edge(t2,estart,estop) ) return false;
	}

	return true;
}


//////////////////////////////////////////////////////////////////////////////
/// @begin is_vertex
///
/// @brief
///
/// @detailed
///
/// @param  tree - [in/out]?
/// @param  pos - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
//////////////////////////////////////////////////////////////////////////////
bool
is_vertex(
	FArray2Da_int tree,
	int const pos
)
{
	tree.dimension( 3, SRange( 0, star ) );

// local
	int estart, estop, elabel;

	for ( int i = 1, ie = num_edges(tree); i <= ie; ++i ) {
		get_edge(tree,i,estart,estop,elabel);
		if ( estart == pos || estop == pos ) {
			return true;
		}
	}
	return false;
}
