// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 14080 $
//  $Date: 2007-04-09 23:55:07 +0300 (Mon, 09 Apr 2007) $
//  $Author: yab $


// Rosetta Headers
#include "void.h"
#include "aaproperties_pack.h"
#include "cst_set.h"
#include "decoystats.h" // for rsd_exposed_sasa
#include "design.h"
#include "FArray_xyz_functions.h"
#include "fullatom_extra_props.h"
#include "fullatom_sasa.h"
#include "fullatom_sasa_ns.h"
#include "interface.h"
#include "ligand.h"
#include "ligand_ns.h"
#include "misc.h"
#include "param.h"
#include "param_aa.h"
#include "pose_ligand.h"
#include "read_paths.h"
#include "RotamerDots.h"
#include "util_vector.h"
#include "void_ns.h"
//KMa phospho_ser
#include "add_pser.h"

// ObjexxFCL Headers
#include <ObjexxFCL/byte.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/Fmath.hh>
#include <ObjexxFCL/formatted.io.hh>

// Numeric Headers
#include <numeric/constants.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <algorithm>
#include <cstdlib>
#include <fstream>
#include <iostream>


//////////////////////////////////////////////////////////////////////////////
/// @begin
///
/// @brief
/// this is mostly a copy of the sasa function in fullatom_sasa.cc
///
/// @detailed
/// the difference is that sasa is calculated for each residue
/// at two different probe radii: 1.4A (water), and 0.5A
///
/// generally speaking, a well packed residue will have a smaller
/// value for the SASA at 0.5A, normalized for the value at 1.4A
/// in other words, whatever part isnt accessible to water would
/// like to be packed against other parts of the protein.
///
/// a set of real proteins was analyzed to see how SASA varies at
/// 0.5 for different values at 1.4 and different residue types
/// this data is contained in the sasa_offsets.txt datafile
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
/// pb
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
calc_sasa_pack_score(
	int const total_residue,
	FArray1D_int const & res,
	FArray1D_int const & res_variant,
	FArray3D_float const & full_coord
)
{
	using namespace aaproperties_pack; // for fullatom_type & natoms
	using namespace fullatom_ex_props; // for aradii & asp
	using namespace fullatom_sasa;
	using namespace interface;
	using namespace ligand;
	//using namespace misc;              // for full_coord
	using namespace param;             // for paramters
	using namespace param_aa;             // for paramters
	using namespace void_ns;
	using namespace numeric::constants::f;

	int in,it,jn,jt,ctr,olp,aphi,theta,point,masknum,aa,lig_int_count;

// do this now, in case we couldnt find the sasa_offsets data-file
	for ( int i = 1; i <= total_residue; ++i ) {
		rsd_sasa_pack_score(i) = 0.0;
	}
	avg_sasa_pack_score = 0.0;

	FArray1D_float ic( 3 );
	FArray1D_float jc( 3 );
	float dist,irad,jrad,fraction,total_sa,expose,irad_base,jrad_base,
	 probe_radius,dist_sq,total_sasa5,total_sasa14,sasa14,
	 sasa5,offset,norm_sasa5;

// save these fields which are initialized the first time through

//pb:: the atom masks!! new index: n_pr_bin
	FArray4D_ubyte atm_masks( nbytes, MAX_ATOM()(), MAX_RES()(), n_pr_bin );
	 // full mask for atoms

	// return if the rosetta_database doesnt contain sasa_offsets.txt
	bool database_file_present = initialize_sasa_pack_arrays();
	if (! database_file_present ) return;


//-----initialize atm_masks to zero
	for ( int pr_bin = 1; pr_bin <= n_pr_bin; ++pr_bin ) {
		for ( int ir = 1; ir <= total_residue; ++ir ) {
			in = res(ir);
			for ( int ia = 1, iae = natoms(in,res_variant(ir)); ia <= iae; ++ia ) {
//cj     first zero mask for all atoms
				for ( int bb = 1, l = atm_masks.index(bb,ia,ir,pr_bin);
				 bb <= nbytes; ++bb, ++l ) {
					atm_masks[ l ] = 0; // atm_masks(bb,ia,ir,pr_bin)
				}
			}               // ia
		}                  // ir
	}

//cj----now do calculations: get the atm_masks by looping over all_atoms x all_atoms

	for ( int ir = 1; ir <= total_residue; ++ir ) {
		in = res(ir);
		for ( int ia = 1, iae = natoms(in,res_variant(ir)); ia <= iae; ++ia ) {

			// store the coordinates and atom type, since we will reuse this a bunch
			for ( int m = 1; m <= 3; ++m ) {
				ic(m) = full_coord(m,ia,ir);
			}

			it = fullatom_type(ia,in,res_variant(ir));
			if ( it != 26 ) {    // jk

				irad_base = aradii(it); // no probe radius included

//cj   could do this more efficiently if found neighbors first???
//jg  loop over jr>ir, to cover all pair-wise interactions
				for ( int jr = ir; jr <= total_residue; ++jr ) {
					jn = res(jr);
					for ( int ja = 1, jae = natoms(jn,res_variant(jr)); ja <= jae; ++ja ) {

						for ( int m = 1; m <= 3; ++m ) {
							jc(m) = full_coord(m,ja,jr);
						}

						jt = fullatom_type(ja,jn,res_variant(jr));
						if ( jt != 26 ) {    // jk

//cj       getting squared distance between atoms (faster than taking sqrt)
							dist_sq = vec_dist2(ic,jc);

							if ( dist_sq <= rsq_min(it,jt,1) ) {

//car ignore all zero distances, either i,j are same atom or it's
//car an invalid pair for some reason...
								if ( dist_sq <= 0.0 ) goto L100;

								jrad_base = aradii(jt); // no probe radius included
								dist = std::sqrt(dist_sq);
								for ( int pr_bin = 1; pr_bin <= n_pr_bin; ++pr_bin ) {

									if ( dist_sq > rsq_min(it,jt,pr_bin) ) goto L100;

									probe_radius = probe_radii(pr_bin);
									irad = irad_base + probe_radius;
									jrad = jrad_base + probe_radius;

									// account for j overlapping i:

									get_overlap(ic,irad,jc,jrad,dist,olp);
									get_orientation(ic,jc,aphi,theta, dist);

									point = angles(aphi,theta);
									masknum = point*100+olp;

									for ( int bb = 1, l = atm_masks.index(bb,ia,ir,pr_bin);
												bb <= nbytes; ++bb, ++l ) { // [ l ] = (bb,ia,ir,pr_bin)
										atm_masks[ l ] = bit_or( atm_masks[ l ], masks(bb,masknum) );
									}

									// account for i overlapping j:

									get_overlap(jc,jrad,ic,irad,dist,olp);
									get_orientation(jc,ic,aphi,theta, dist);

									point = angles(aphi,theta);
									masknum = point*100+olp;

									for ( int bb = 1, l = atm_masks.index(bb,ja,jr,pr_bin);
												bb <= nbytes; ++bb, ++l ) { // [ l ] = (bb,ja,jr,pr_bin)
										atm_masks[ l ] = bit_or( atm_masks[ l ], masks(bb,masknum) );
									}
								}      // pr_bin
							} else if ( dist_sq > 289.0 ) {
								// 289 = 17A^2 = 10A + 2 (1.4 + 2.1)
								goto L110; // this residue is too far away
								// go to next residue j
							}         // dij_sq <= rij_sq
L100:;
						}   // jk if not water
					}            // ja
L110:;
					}               // jr

//mj if ligand flag sum over all heteroatoms
				if ( get_ligand_flag() && int_res(ir) ) {
					for ( int ja = 0; ja < int(ligand::ligand_one->atom_vector.size()); ++ja ) {

   					jt = ligand::ligand_one->atom_vector[ja]->get_ros_atom_type();
						if ( jt > MAX_REALTYPES ) goto L123;

//						for ( int m = 1; m <= 3; ++m ) {
							copy_to_FArray(ligand::ligand_one->atom_vector[ja]->get_coordinates(),jc);

//						}

//cj       getting squared distance between atoms (faster than taking sqrt)
						dist_sq = vec_dist2(ic,jc);

						if ( dist_sq <= rsq_min(it,jt,1) ) {

//car ignore all zero distances, either i,j are same atom or it's
//car an invalid pair for some reason...
							if ( dist_sq <= 0.0 ) goto L123;

							jrad_base = aradii(jt); // no probe radius included
							dist = std::sqrt(dist_sq);
							for ( int pr_bin = 1; pr_bin <= n_pr_bin; ++pr_bin ) {

								if ( dist_sq > rsq_min(it,jt,pr_bin) ) goto L123;

								probe_radius = probe_radii(pr_bin);
								irad = irad_base + probe_radius;
								jrad = jrad_base + probe_radius;

								// account for j overlapping i:

								get_overlap(ic,irad,jc,jrad,dist,olp);
								get_orientation(ic,jc,aphi,theta, dist);

								point = angles(aphi,theta);
								masknum = point*100+olp;

								for ( int bb = 1, l = atm_masks.index(bb,ia,ir,pr_bin);
											bb <= nbytes; ++bb, ++l ) { // [ l ] = (bb,ia,ir,pr_bin)
									atm_masks[ l ] = bit_or( atm_masks[ l ], masks(bb,masknum) );
								}
							}      // pr_bin
						}
L123:;
					}            // ja
				}
//mj end sum over all heteroatoms

			}  // jk if not water
		}                  // ia
	}                     // ir


//-----calculate the per-residue sasa

	float const four_pi = 4.0f * pi;
	for ( int pr_bin = 1; pr_bin <= n_pr_bin; ++pr_bin ) {
		probe_radius = probe_radii(pr_bin);
		for ( int ir = 1; ir <= total_residue; ++ir ) {
			rsd_sasa(pr_bin,ir) = 0.0;
			in = res(ir);
			for ( int ia = 1, iae = natoms(in,res_variant(ir)); ia <= iae; ++ia ) {

				it = fullatom_type(ia,in,res_variant(ir));
//				irad = aradii(it) + 1.4;
				irad = aradii(it); //!!!!!!!!! dont include probe !!!!!!!!!!!!!

//cj       to get SASA:
//cj       - count the number of 1's
//cj       - figure fraction that they are
//cj       - multiply by 4*pi*r_sqared
				ctr = 0;
				for ( int bb = 1, l = atm_masks.index(bb,ia,ir,pr_bin);
				 bb <= nbytes; ++bb, ++l ) {
					ctr += bit_count[atm_masks[ l ]]; // atm_masks(bb,ia,ir,pr_bin)
				}

				fraction = static_cast< float >( ctr ) / maskbits;
				total_sa = four_pi * ( irad * irad );

				expose = ( 1.0f - fraction ) * total_sa;
				rsd_sasa(pr_bin,ir) += expose;
				atom_sasa(pr_bin,ia,ir) = expose;

			}               // ia
		}                  // ir
	}                     // pr_bin

	//lin atomic SASA array for ligaa
	if( cst_set_ns::debug_output ) {
		using namespace enable_ligaa_ns;
		for ( int ii = 1; ii <= total_residue; ++ii ) {
			if( is_ligand( res(ii) ) ) {
				for( int jj=1, je=natoms(res(ii),1); jj<=je; jj++ ) {
					sasa14 = atom_sasa(pr_bin_14,jj,ii);
					sasa5 = atom_sasa(pr_bin_5,jj,ii);
					norm_sasa5 = sasa5 - sasa14;
					fill_ligaa_sasa_arrays( jj, res(ii), sasa14, sasa5, norm_sasa5 );
				}
			}
		}
	}

//pb now calculate the normalized surface areas by comparing to the database averages

	total_sasa5  = 0.0; // total protein SASA at probe radius of 0.5
	total_sasa14 = 0.0; // total protein SASA at probe radius of 1.4
	avg_sasa_pack_score = 0.0; // sasa_pack_score averaged over all residues
	ligand::ligand_one->lig_sasE = 0.0; // sasa_pack_score averaged over int residues
	lig_int_count = 0;

	for ( int i = 1; i <= total_residue; ++i ) {
		aa = res(i);
    if ( !is_protein(aa) && !is_ligand(aa) ) goto L999; // escape for non-protein

		sasa14 = rsd_sasa(pr_bin_14,i);
		sasa5 = rsd_sasa(pr_bin_5,i);

//        Look up the average sasa5 for this range of sasa14 given the amino acid type
		if ( sasa14 < 0.0 ) {
			sasa14 = 0.0;
		} else if ( sasa14 > 999.0 ) {
			std::cout << "big sasa14!" << SS( sasa14 ) << std::endl;
			sasa14 = 999.0;
		}

		offset = -1.0;
		for ( int j = 1; j <= max_sasa_offset_bins; ++j ) { //Objexx:SGM Unassigned elements accessed assuming they are zero
			if ( is_protein(aa) && sasa14 >= sasa_offset(1,j,aa) && sasa14 < sasa_offset(2,j,aa) ) {
				offset = sasa_offset(3,j,aa);
			} else if( !is_protein(aa) ) {
				offset = 0.0;
			}
		}
		if ( offset == -1.0 ) { // this should never happen, tell phil b if it does
			std::cout << "sasa14 out of bounds!!" << SS( sasa14 ) << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		norm_sasa5 = sasa5 - sasa14 - offset;

		total_sasa5 += sasa5;
		total_sasa14 += sasa14;

		rsd_sasa_pack_score(i) = norm_sasa5;
		avg_sasa_pack_score += norm_sasa5;

		if ( get_ligand_flag() && int_res( i) ) {
			ligand::ligand_one->lig_sasE += norm_sasa5;
			++lig_int_count;
		}

		L999:; // escape for non-protein aa's
	}

	if ( total_residue > 0 ) avg_sasa_pack_score /= total_residue;
	if ( get_ligand_flag() && lig_int_count > 0 ) ligand::ligand_one->lig_sasE /= lig_int_count;

	// jk Call this here, to avoid problem that design mode calls "make_pdb" instead of "output_decoy"
	calc_sasa_prob();

}

bool
initialize_sasa_pack_arrays()
{
	using namespace aaproperties_pack; // for fullatom_type & natoms
	using namespace fullatom_ex_props; // for aradii & asp
	using namespace fullatom_sasa;
	using namespace interface;
	using namespace ligand;
	using namespace misc;              // for full_coord
	using namespace param;             // for paramters
	using namespace param_aa;             // for paramters
	using namespace void_ns;
	using namespace numeric::constants::f;

	static bool sasa_pack_init = { false };
	static bool file_found = { false };

	int rsd_i,last_rsd_i,total_bins,bin_count;

	FArray1D_float ic( 3 );
	FArray1D_float jc( 3 );
	float bin_start,bin_stop,offset;

	char rsd_c;
	bool fail;

	if ( ! sasa_pack_init ) { // read datafile, initialize arrays
		sasa_pack_init = true;

		for ( int pr_bin = 1; pr_bin <= n_pr_bin; ++pr_bin ) {
			probe_calc_min_rsq( rsq_min(1,1,pr_bin), probe_radii(pr_bin) );
		}

//pb   read the sasa_offset file
		utility::io::izstream & sasa_offsets_stream(
		 try_to_open_data_file( "sasa_offsets.txt", fail ) );

		if ( fail ) {
			file_found = false;
			sasa_offsets_stream.clear();
			return file_found;
		} else {
			file_found = true;
		}

		sasa_offsets_stream >> total_bins >> skip;
		last_rsd_i = -1;
		bin_count = 0; // just to avoid compiler warning
		sasa_offset = 0.0; //Objexx:SGM Unassigned elements are accessed below
		for ( int i = 1; i <= total_bins; ++i ) {
			sasa_offsets_stream >> rsd_c >> rsd_i >> bin_start >> bin_stop >>
			 offset >> skip;
//			std::cout << SS( i ) << SS( total_bins ) << SS( rsd_c) << SS( rsd_i ) <<
//			 SS( bin_start ) << SS( bin_stop ) << SS( offset ) << std::endl;
			if ( rsd_i != last_rsd_i ) {
				last_rsd_i = rsd_i;
				bin_count = 0;
//				std::cout << "rebin_start to 0.0:" << SS( bin_start ) << std::endl;
				bin_start = 0.0;
			}
			if ( rsd_c != aa_name1(rsd_i) ) { // this shouldnt happen
				std::cout << "aa numbering problem!" << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
			++bin_count;
			if ( bin_count > max_sasa_offset_bins ) {
				std::cout << "too many offset bins!" << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
			sasa_offset(1,bin_count,rsd_i) = bin_start;
			sasa_offset(2,bin_count,rsd_i) = bin_stop;
			sasa_offset(3,bin_count,rsd_i) = offset;
//KMa phospho_ser
	 if ( add_pser() )
		   if (rsd_i==param_aa::aa_ser)
			{
				sasa_offset(1,bin_count,param_aa::aa_sep) = bin_start;
				sasa_offset(2,bin_count,param_aa::aa_sep) = bin_stop;
				sasa_offset(3,bin_count,param_aa::aa_sep) = offset;
			}
		}
		sasa_offsets_stream.close();
		sasa_offsets_stream.clear();
	}

	return file_found;
}


//////////////////////////////////////////////////////////////////////////////
/// @begin retrieve_sasa_pack_score
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
retrieve_sasa_pack_score()
{
	using namespace void_ns;

	return avg_sasa_pack_score;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin retrieve_sub_sasa_pack_score
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
retrieve_sub_sasa_pack_score(
	FArray1Da_bool allow_region
)
{
	using namespace void_ns;
	using namespace param; // for MAX_RES
	using namespace misc;
	using namespace design;

	allow_region.dimension( MAX_RES() );

	float sasa_pack = 0.0;
	int count = 0;

	for ( int i = 1; i <= total_residue; ++i ) {
		if ( allow_region(i) ) {
			++count;
			sasa_pack += rsd_sasa_pack_score(i);
			//if ( debug_output ) std::cout<<"DEBUG: allow_repack at pos "<<i<<" sasa_pack "<<  rsd_sasa_pack_score(i) <<std::endl;
		}
	}

	return ( count == 0 ? 0.0f : sasa_pack / count );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin probe_calc_min_rsq
///
/// @brief
/// just like calc_min_rsq in fullatom_sasa.cc, but the radius gets passed in
///
/// @detailed
/// this function pre-tabulates the minimun squared-distances between
/// atoms of all types.   this should speed up the calculation
/// tremendously, by allowing us not to take square roots in the main
/// loop of calc_approx_sasa when not necessary.  jjg 2/4/2
///
/// @param  rsq_min - [in/out]? -
/// @param[in] radius -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
probe_calc_min_rsq(
	FArray2Da_float rsq_min,
	float const radius // input
)
{
	using namespace fullatom_ex_props; // for aradii
	using namespace param; // for MAX_ATOMTYPES

	rsq_min.dimension( MAX_ATOMTYPES(), MAX_ATOMTYPES() );

	for ( int i = 1, e = MAX_ATOMTYPES(); i <= e; ++i ) {
		for ( int j = i; j <= e; ++j ) {
			rsq_min(i,j) = rsq_min(j,i) =
			 square( ( aradii(i) + radius ) + ( aradii(j) + radius ) );
		}
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin initialize_sasa_prob_arrays
///
/// @brief read in info from sasa_prob_cdf.txt for sasa prob calculations
/////////////////////////////////////////////////////////////////////////////////
bool
initialize_sasa_prob_arrays()
{
	using namespace misc;
	using namespace param;
	using namespace param_aa;
	using namespace std;
	using namespace void_ns;

	static bool data_file_found;
	static bool sasa_prob_init = { false };
	// read sasaprob data file if necessary
	if ( !sasa_prob_init ) {
		sasa_prob_init = true;

		bool fail;
		utility::io::izstream & sasaprobin( try_to_open_data_file( "sasa_prob_cdf.txt", fail ) );
		if ( fail ) {
			data_file_found = false;
			cout << "WARNING: can't find sasa_prob_cdf.txt data file" << endl;
			sasaprobin.clear();
			return data_file_found;
		} else {
			data_file_found = true;
		}

		int nres,nlines,nrow,resnum;
		float f;
		string res;
		sasaprobin >> nres;
		for ( int ii = 1; ii <= nres; ++ii ) {
			sasaprobin >> resnum >> res >> nlines >> nrow;
			for ( int jj = 1; jj <= nlines; ++jj ) {
				for ( int kk = 1; kk <= nrow; ++kk ) {
					sasaprobin >> f;
					sasaprob_cdf(ii,kk,jj) = f;
					//KMa phospho_ser
				 	if ( add_pser() ){
						if (ii==param_aa::aa_ser){
							sasaprob_cdf(param_aa::aa_sep,kk,jj)=sasaprob_cdf(ii,kk,jj);
						}
					}
				}
			}
		}

		sasaprobin.close();
		sasaprobin.clear();
	}
	return data_file_found;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin calc_sasa_prob
///
/// @brief   assumes that calc_sasa_pack_score already called successfully
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
calc_sasa_prob()
{
	using namespace misc;
	using namespace param;
	using namespace param_aa;
	using namespace std;
	//using namespace template_pack; // for aan
	using namespace void_ns;

	for ( int i = 1; i <= total_residue; ++i ) {
		rsd_sasaprob(i) = -1.0;
	}
	avg_sasaprob = -1.0;

	bool database_file_present = initialize_sasa_prob_arrays();
	if (! database_file_present) {
		std::cout << "Could not read SASA prob info from rosetta database" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	if (! database_file_present) {
//	std::cerr << "!!!!!";
		return;
	}

	// sasa fraction
	FArray2D_float atom_sasa14( MAX_ATOM()(), MAX_RES()() );
	FArray1D_float tmp_rsd_sasa14( MAX_RES()() );
	calc_per_atom_sasa( atom_sasa14, tmp_rsd_sasa14, 1.4f, false, false );

 	for ( int ii = 1; ii <= total_residue; ++ii ) {
 		int const aa = res(ii);
//KMa phospho_ser
		if (  aa > MAX_AA_PLUS()  ) continue; // escape for non-protein

		float sasaprob = 0.0;
		float const sasa5  = void_ns::void_common::rsd_sasa(pr_bin_5,ii);
		float const sasa14 = void_ns::void_common::rsd_sasa(pr_bin_14,ii);

		sasaprob = calc_residue_sasa_prob(aa, sasa5, sasa14, tmp_rsd_sasa14(ii));

		rsd_sasaprob(ii) = sasaprob;
		//std::cerr << "sasaprob: " << ii << " " << sasaprob << std::endl;
		avg_sasaprob += sasaprob;
	}

	avg_sasaprob = avg_sasaprob / total_residue;

}

//////////////////////////////////////////////////////////////////////////////
/// @begin retrieve_sasa_prob
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
retrieve_rsd_sasa_prob( int const res )
{
	using namespace void_ns;
	return rsd_sasaprob( res );
}


float
retrieve_avg_sasa_prob()
{
	using namespace void_ns;
	return avg_sasaprob;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin calc_residue_sasa_prob
///
/// @brief get sasa pack prob given aa type, sasa at 0.5 A, sasa at 1.4 A
///        and sasa pack score at 1.4 A
////////////////////////////////////////////////////////////////////////////////
float
calc_residue_sasa_prob(
	int this_aa,
	float sasa5,
	float sasa14, //apl this variable is the molecular surface area accessible to solvent (MSAAS)
	float sasa14_pack_score //apl <- this variable appears to be the solvent accessible surface area (NOT MSAAS, and NOT sasapack)!
){
	using namespace misc;
	using namespace param;
	using namespace param_aa;
	using namespace std;
	using namespace void_ns;

	float sasaprob = 0.0;

	float const sasafrac = std::max( 0.0f,
	 std::min( 1.0f, sasa14_pack_score / rsd_exposed_sasa( this_aa ) ) );
	int const sasadiff = std::min( std::max( static_cast< int >( sasa5 - sasa14 ), 1 ), 75 );

	//whs binning probably shouldn't be hard coded...
	int sasabin = 1;
	float binpos = 0.0; // how far into the bin are we
	if ( sasafrac == 0.0 ) {
		sasabin = 1;
		binpos  = 1.0;
		sasaprob = sasaprob_cdf( this_aa, sasabin, sasadiff );
	} else if ( sasafrac <= 0.02 ) {
		sasabin = 2;
		binpos = ( sasafrac - 0.00 ) / ( 0.02 - 0.00 );
	} else if ( sasafrac <= 0.05 ) {
		sasabin = 3;
		binpos = ( sasafrac - 0.02 ) / ( 0.05 - 0.02 );
	} else if ( sasafrac <= 0.09 ) {
		sasabin = 4;
		binpos = ( sasafrac - 0.05 ) / ( 0.09 - 0.05 );
	} else if ( sasafrac <= 0.15 ) {
		sasabin = 5;
		binpos = ( sasafrac - 0.09 ) / ( 0.15 - 0.09 );
	} else if ( sasafrac <= 0.22 ) {
		sasabin = 6;
		binpos = ( sasafrac - 0.15 ) / ( 0.22 - 0.15 );
	} else if ( sasafrac <= 0.30 ) {
		sasabin = 7;
		binpos = ( sasafrac - 0.22 ) / ( 0.30 - 0.22 );
	} else if ( sasafrac <= 0.45 ) {
		sasabin = 8;
		binpos = ( sasafrac - 0.30 ) / ( 0.45 - 0.30 );
	} else if ( sasafrac <= 0.65 ) {
		sasabin = 9;
		binpos = ( sasafrac - 0.45 ) / ( 0.60 - 0.45 );
	} else if ( sasafrac <= 1.00 ) {
		sasabin = 10;
		binpos = ( sasafrac - 0.60 ) / ( 1.00 - 0.60 );
	} else {
		cout << "ERROR void.cc bad sasafrac!" << endl;
	}
	sasaprob  = sasaprob_cdf( this_aa, sasabin, sasadiff ) * binpos;
	if ( sasabin > 1 ) {
		sasaprob += sasaprob_cdf( this_aa, sasabin-1, sasadiff ) * ( 1.0 - binpos );
	}

	return sasaprob;
}

/////////////////////////////////////////////////////////////////////////////
///@begin get_res_res_overlap
///
///@brief
///
///@detailed
//////////////////////////////////////////////////////////////////////////////
void
get_res_res_overlap(
  pack::RotamerCoords const & res1,
  pack::RotamerCoords const & res2,
  FArray2DB< ubyte > & res1_covered_by_res2_small,
  FArray2DB< ubyte > & res1_covered_by_res2_large,
  FArray2DB< ubyte > & res2_covered_by_res1_small,
  FArray2DB< ubyte > & res2_covered_by_res1_large,
  FArray1DB_bool & res1_atoms_overlap_at_all,
  FArray1DB_bool & res2_atoms_overlap_at_all
){

	using namespace fullatom_sasa;

  res1_covered_by_res2_small = static_cast< ubyte > ( 0 );
  res1_covered_by_res2_large = static_cast< ubyte > ( 0 );
  res2_covered_by_res1_small = static_cast< ubyte > ( 0 );
  res2_covered_by_res1_large = static_cast< ubyte > ( 0 );
  res1_atoms_overlap_at_all = false;
  res2_atoms_overlap_at_all = false;

  float square_distance;
  for (int ii = 1; ii <= res1.get_num_atoms(); ++ii)
  {
	float ii_atom_radius = res1.get_atom_radius( ii );
    for (int jj = 1; jj <= res2.get_num_atoms(); ++jj)
    {
	  float jj_subtree_radius = res2.get_subtree_interaction_sphere_radius( jj );
      bool overlap = get_2_way_atom_atom_coverage(
			res1.get_atom_coords(ii), res1.get_atom_type( ii ),
			res2.get_atom_coords(jj), res2.get_atom_type( jj ),
			res1_covered_by_res2_small(1, ii),
			res2_covered_by_res1_small(1, jj),
			res1_covered_by_res2_large(1, ii),
			res2_covered_by_res1_large(1, jj),
			square_distance
      );

      res1_atoms_overlap_at_all( ii ) |= overlap;
      res2_atoms_overlap_at_all( jj ) |= overlap;

      if (square_distance > (ii_atom_radius + jj_subtree_radius)*(ii_atom_radius + jj_subtree_radius) )
      {
        break;
      }

    }
  }
}

void
get_res_res_overlap(
	pack::RotamerCoords const & res1,
	pack::RotamerCoords const & res2,
	FArray2DB< ubyte > & res1_covered_by_res2_small,
	FArray2DB< ubyte > & res1_covered_by_res2_large,
	FArray2DB< ubyte > & res2_covered_by_res1_small,
	FArray2DB< ubyte > & res2_covered_by_res1_large,
	FArray3DB< short > & overlap_masks,
	int num_res2_atoms_to_ignore,
	FArray1DB_bool & res1_atoms_overlap_at_all,
	FArray1DB_bool & res2_atoms_overlap_at_all
)
{
	using namespace fullatom_sasa;

	float square_distance;
	for (int ii = 1; ii <= res1.get_num_atoms(); ++ii)
	{
		float ii_atom_radius = res1.get_atom_radius( ii );
		bool subtree_worth_examining = true;
		for (int jj = num_res2_atoms_to_ignore + 1; jj <= res2.get_num_atoms(); ++jj)
		{
			float jj_subtree_radius = res2.get_subtree_interaction_sphere_radius( jj );

			if (subtree_worth_examining)
			{
				bool overlap = get_2_way_atom_atom_coverage(
					res1.get_atom_coords(ii), res1.get_atom_type( ii ),
					res2.get_atom_coords(jj), res2.get_atom_type( jj ),
					res1_covered_by_res2_small(1, ii),
					res2_covered_by_res1_small(1, jj),
					res1_covered_by_res2_large(1, ii),
					res2_covered_by_res1_large(1, jj),
					overlap_masks(1, jj, ii),
					square_distance
					);

				res1_atoms_overlap_at_all( ii ) |= overlap;
				res2_atoms_overlap_at_all( jj ) |= overlap;

				if (square_distance > (ii_atom_radius + jj_subtree_radius)*(ii_atom_radius + jj_subtree_radius) )
				{
					subtree_worth_examining = false;
				}
			}
			else
			{
				for (int kk = 1, li = overlap_masks.index(1, jj, ii); kk <= 4; ++kk, ++li)
				{
					overlap_masks[ li ] = 0;
				}
			}
		}
	}
}

void
get_residue_ligand_overlap
(
	pack::RotamerCoords const & res,
	FArray2DB< ubyte > & res_covered_by_lig_small,
	FArray2DB< ubyte > & res_covered_by_lig_large,
	FArray1DB_bool & res1_atoms_overlap_at_all,
	Ligand & ligand
)
{

	using namespace fullatom_ex_props; // for aradii & asp

	FArray2D_float lig_coords;
	FArray1D_int lig_atom_types;
	FArray1D_float lig_radii;

	ligand.get_FArray2D_of_coordinates( ligand.atom_vector, lig_coords );
	ligand.get_FArray1D_of_atom_type( ligand.atom_vector, lig_atom_types );
	int const lig_num_atoms = lig_atom_types.size1();
	lig_radii.dimension( lig_num_atoms );
	for (int ii = 1; ii <= lig_num_atoms; ++ii)
	{
		lig_radii = aradii( lig_atom_types( ii ) );
	}

	float square_distance;
	for (int ii = 1; ii <= lig_num_atoms; ++ii)
	{
		float ii_atom_radius = lig_radii( ii );
		for (int jj = 1; jj <= res.get_num_atoms(); ++jj)
		{
			float const jj_subtree_radius = res.get_subtree_interaction_sphere_radius( jj );

			bool overlaps = get_atom_coverage(
				res.get_atom_coords(jj), res.get_atom_type( jj ),
				FArray1Da_float(lig_coords(1, ii), 3), lig_atom_types( ii ),
				res_covered_by_lig_small(1, jj),
				res_covered_by_lig_large(1, jj),
				square_distance
			);

			res1_atoms_overlap_at_all( jj ) |= overlaps;

			if (square_distance > (ii_atom_radius + jj_subtree_radius)*(ii_atom_radius + jj_subtree_radius) )
			{
				break;
			}
		}
	}

}



void
get_residue_self_coverage(
	pack::RotamerCoords const & res,
	FArray2DB< ubyte > & res_self_overlap_small,
	FArray2DB< ubyte > & res_self_overlap_large
)
{

	using namespace fullatom_sasa;

	res_self_overlap_small = static_cast< ubyte > ( 0 );
	res_self_overlap_large = static_cast< ubyte > ( 0 );

	float sqr_dist;
	for (int ii = 1; ii <= res.get_num_atoms(); ++ii){
		for (int jj = ii+1; jj <= res.get_num_atoms(); ++jj){
			get_2_way_atom_atom_coverage(
				res.get_atom_coords(ii), res.get_atom_type( ii ),
				res.get_atom_coords(jj), res.get_atom_type( jj ),
				res_self_overlap_small(1, ii),
				res_self_overlap_small(1, jj),
				res_self_overlap_large(1, ii),
				res_self_overlap_large(1, jj),
				sqr_dist
			);
		}
	}
}

//compute the portion of the two spheres surrounding atom1 contained
// within the two spheres surrounding atom2.
bool
get_atom_coverage(
	FArray1DB_float const & at1_coord,
	int at1_atomtype,
	FArray1DB_float const & at2_coord,
	int at2_atomtype,
	FArray1Da_ubyte at1_small_sphere_covered,
	FArray1Da_ubyte at1_large_sphere_covered,
	float & square_distance
)
{
	using namespace aaproperties_pack; // for fullatom_type & natoms
	using namespace fullatom_ex_props; // for aradii & asp
	using namespace fullatom_sasa;
	using namespace interface;
	using namespace ligand;
	using namespace misc;              // for full_coord
	using namespace param;             // for paramters
	using namespace param_aa;             // for paramters
	using namespace void_ns;
	using namespace numeric::constants::f;

	at1_small_sphere_covered.dimension( nbytes );
	at1_large_sphere_covered.dimension( nbytes );

	int olp;
	int aphi;
	int theta;
	int point1;
	int masknum;
	float at1_base_radius,at2_base_radius,dist,at1_radius,at2_radius;

	//get distance
	at1_base_radius = aradii(at1_atomtype);
	at2_base_radius = aradii(at2_atomtype);
	square_distance = vec_dist2(at1_coord,at2_coord);

	//calculate masks: large probe
	at1_radius = at1_base_radius + probe_radius_large;
	at2_radius = at2_base_radius + probe_radius_large;

	//exit if large probe radii do not touch
	if ( square_distance > (at1_radius + at2_radius)*(at1_radius + at2_radius) ){
		return false;
	}

	dist = std::sqrt(square_distance);

	//large atom1 overlap
	get_overlap(at1_coord,at1_radius,at2_coord,at2_radius,dist,olp);
	get_orientation(at1_coord,at2_coord,aphi,theta, dist);

	point1 = angles(aphi,theta);
	masknum = point1*100+olp;
	for ( int bb = 1; bb <= nbytes; ++bb ){
		at1_large_sphere_covered(bb) |= masks(bb,masknum);
	}

	//small probe
	at1_radius = at1_base_radius + probe_radius_small;
	at2_radius = at2_base_radius + probe_radius_small;

	if ( dist < (at1_radius + at2_radius))
	{

		//small atom1 overlap
		get_overlap(at1_coord,at1_radius,at2_coord,at2_radius,dist,olp);
		masknum = point1*100+olp;
		for ( int bb = 1; bb <= nbytes; ++bb ){
			at1_small_sphere_covered(bb) |= masks(bb,masknum);
		}

	}
	return true;

}

//////////////////////////////////////////////////////////////////////////////
///@begin
///
///@brief
///returns false if the two spheres do not overlap at all
///
///@detailed
///
///@remarks
///
///@authors
///
///@last_modified
//////////////////////////////////////////////////////////////////////////////
bool
get_2_way_atom_atom_coverage(
  FArray1DB_float const & at1_coord,
  int at1_atomtype,
  FArray1DB_float const & at2_coord,
  int at2_atomtype,
  FArray1Da_ubyte at1_small_sphere_covered,
  FArray1Da_ubyte at2_small_sphere_covered,
  FArray1Da_ubyte at1_large_sphere_covered,
  FArray1Da_ubyte at2_large_sphere_covered,
  float & square_distance
){

	using namespace aaproperties_pack; // for fullatom_type & natoms
	using namespace fullatom_ex_props; // for aradii & asp
	using namespace fullatom_sasa;
	using namespace interface;
	using namespace ligand;
	using namespace misc;              // for full_coord
	using namespace param;             // for paramters
	using namespace param_aa;             // for paramters
	using namespace void_ns;
	using namespace numeric::constants::f;

	at1_small_sphere_covered.dimension( nbytes );
	at2_small_sphere_covered.dimension( nbytes );
	at1_large_sphere_covered.dimension( nbytes );
	at2_large_sphere_covered.dimension( nbytes );

	int olp;
	int aphi_1_2, aphi_2_1;
	int theta_1_2, theta_2_1;
	int point1,point2;
	int masknum;
	float at1_base_radius,at2_base_radius,dist,at1_radius,at2_radius;

	//get distance
	at1_base_radius = aradii(at1_atomtype);
	at2_base_radius = aradii(at2_atomtype);
	square_distance = vec_dist2(at1_coord,at2_coord);

	//calculate masks: large probe
	at1_radius = at1_base_radius + probe_radius_large;
	at2_radius = at2_base_radius + probe_radius_large;

	//exit if large probe radii do not touch
	if ( square_distance > (at1_radius + at2_radius)*(at1_radius + at2_radius) ){
		return false;
	}

	dist = std::sqrt(square_distance);

	//large atom1 overlap
	get_overlap(at1_coord,at1_radius,at2_coord,at2_radius,dist,olp);
	get_2way_orientation(at1_coord,at2_coord,aphi_1_2,theta_1_2, aphi_2_1, theta_2_1, dist);


	point1 = angles(aphi_1_2,theta_1_2);
	masknum = point1*100+olp;
	for ( int bb = 1; bb <= nbytes; ++bb ){
		at1_large_sphere_covered(bb) |= masks(bb,masknum);
	}

	//large atom2 overlap
	get_overlap(at2_coord,at2_radius,at1_coord,at1_radius,dist,olp);
	point2 = angles(aphi_2_1,theta_2_1);
	masknum = point2*100+olp;
	for ( int bb = 1; bb <= nbytes; ++bb ){
		at2_large_sphere_covered(bb) |= masks(bb,masknum);
	}

	//small probe
	at1_radius = at1_base_radius + probe_radius_small;
	at2_radius = at2_base_radius + probe_radius_small;

	if ( dist < (at1_radius + at2_radius))
	{

		//small atom1 overlap
		get_overlap(at1_coord,at1_radius,at2_coord,at2_radius,dist,olp);
		masknum = point1*100+olp;
		for ( int bb = 1; bb <= nbytes; ++bb ){
			at1_small_sphere_covered(bb) |= masks(bb,masknum);
		}

		//small atom2 overlap
		get_overlap(at2_coord,at2_radius,at1_coord,at1_radius,dist,olp);
		masknum = point2*100+olp;
		for ( int bb = 1; bb <= nbytes; ++bb ){
			at2_small_sphere_covered(bb) |= masks(bb,masknum);
		}
	}

	return true;

}

bool
get_2_way_atom_atom_coverage(
  FArray1DB_float const & at1_coord,
  int at1_atomtype,
  FArray1DB_float const & at2_coord,
  int at2_atomtype,
  FArray1Da_ubyte at1_small_sphere_covered,
  FArray1Da_ubyte at2_small_sphere_covered,
  FArray1Da_ubyte at1_large_sphere_covered,
  FArray1Da_ubyte at2_large_sphere_covered,
  FArray1Da_short overlap_masks,
  float & square_distance
){

	using namespace aaproperties_pack; // for fullatom_type & natoms
	using namespace fullatom_ex_props; // for aradii & asp
	using namespace fullatom_sasa;
	using namespace interface;
	using namespace ligand;
	using namespace misc;              // for full_coord
	using namespace param;             // for paramters
	using namespace param_aa;             // for paramters
	using namespace void_ns;
	using namespace numeric::constants::f;

	at1_small_sphere_covered.dimension( nbytes );
	at2_small_sphere_covered.dimension( nbytes );
	at1_large_sphere_covered.dimension( nbytes );
	at2_large_sphere_covered.dimension( nbytes );
	overlap_masks.dimension( 4 );
	overlap_masks = 0;

	int olp;
	int aphi_1_2, aphi_2_1;
	int theta_1_2, theta_2_1;
	int point1,point2;
	int masknum;
	float at1_base_radius,at2_base_radius,dist,at1_radius,at2_radius;

	//get distance
	at1_base_radius = aradii(at1_atomtype);
	at2_base_radius = aradii(at2_atomtype);
	square_distance = vec_dist2(at1_coord,at2_coord);

	//calculate masks: large probe
	at1_radius = at1_base_radius + probe_radius_large;
	at2_radius = at2_base_radius + probe_radius_large;

	//exit if large probe radii do not touch
	if ( square_distance > (at1_radius + at2_radius)*(at1_radius + at2_radius) ){
		return false;
	}

	dist = std::sqrt(square_distance);

	//large atom1 overlap
	get_overlap(at1_coord,at1_radius,at2_coord,at2_radius,dist,olp);
	get_2way_orientation(at1_coord,at2_coord,aphi_1_2,theta_1_2, aphi_2_1, theta_2_1, dist);


	point1 = angles(aphi_1_2,theta_1_2);
	masknum = point1*100+olp;
	overlap_masks(1) = masknum;
	for ( int bb = 1; bb <= nbytes; ++bb ){
		at1_large_sphere_covered(bb) |= masks(bb,masknum);
	}

	//large atom2 overlap
	get_overlap(at2_coord,at2_radius,at1_coord,at1_radius,dist,olp);
	point2 = angles(aphi_2_1,theta_2_1);
	masknum = point2*100+olp;
	overlap_masks(2) = masknum;
	for ( int bb = 1; bb <= nbytes; ++bb ){
		at2_large_sphere_covered(bb) |= masks(bb,masknum);
	}

	//small probe
	at1_radius = at1_base_radius + probe_radius_small;
	at2_radius = at2_base_radius + probe_radius_small;

	if ( dist < (at1_radius + at2_radius))
	{

		//small atom1 overlap
		get_overlap(at1_coord,at1_radius,at2_coord,at2_radius,dist,olp);
		masknum = point1*100+olp;
		overlap_masks(3) = masknum;
		for ( int bb = 1; bb <= nbytes; ++bb ){
			at1_small_sphere_covered(bb) |= masks(bb,masknum);
		}

		//small atom2 overlap
		get_overlap(at2_coord,at2_radius,at1_coord,at1_radius,dist,olp);
		masknum = point2*100+olp;
		overlap_masks(4) = masknum;
		for ( int bb = 1; bb <= nbytes; ++bb ){
			at2_small_sphere_covered(bb) |= masks(bb,masknum);
		}
	}

	return true;

}

