// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 18466 $
//  $Date: 2007-11-16 23:05:47 +0200 (Fri, 16 Nov 2007) $
//  $Author: yab $


// Rosetta Headers
#include "wobble.h"
#include "after_opts.h" // for command line flag to replace chuck with smooth.
#include "angles.h"
#include "cenlist.h"
#include "chuck.h"
#include "current_pose.h" // for moves_check_current_pose
#include "elliptic_msd.h"
#include "fragments.h"
#include "gunn.h" // for command line flag to replace chuck with smooth.
#include "jumping_loops.h" // for add_fast_wobble_CCD
#include "loops.h"
#include "maps.h"
#include "minimize.h"
#include "misc.h"
#include "namespace_cold.h"
#include "param.h"
#include "param_aa.h"
#include "pose.h"
#include "ramachandran.h"
#include "random_numbers.h"
#include "recover.h"
#include "refold.h"
#include "rotate.h"
#include "runlevel.h"
#include "smallmove.h"
#include "util_vector.h"
#include "vdw.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/Fmath.hh>
#include <ObjexxFCL/formatted.o.hh>

// Numeric Headers
#include <numeric/conversions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <cstdlib>
#include <iostream>


//namespaces
namespace old_transform {
	FArray2D_float A2( 3, 3 );
	FArray1D_float U2( 3 );
}
namespace splint_elipse {
	FArray1D_float EL( 10 );
	FArray1D_float EL2( 10 );
	FArray2D_float C_EL( 3, 3 ); // the inertial cross moments
}
namespace splint_piston {
	float com_weight = { 1.0 };
}

namespace wobble_splint {
	using namespace param;
	int const MAX_WOBBLE = { 10 };
	int splint_length;
	int const MAX_SPLINT = { 10 };
	FArray1D_float splint_phi( MAX_SPLINT );
	FArray1D_float splint_psi( MAX_SPLINT );
	FArray1D_float splint_omega( MAX_SPLINT );
	FArray1D_char splint_ss( MAX_SPLINT );
	FArray1D_int splint_res( MAX_SPLINT );
	FArray3D_float splint_xyz(3,MAX_POS,MAX_SPLINT+1);
	FArray1D_int splint_angle_map( MAX_WOBBLE );
	int splint_map_length;
}

namespace wobble_torsions {
	int const TORSIONS_PER_RESIDUE = 2;
	bool const vary_psi = (TORSIONS_PER_RESIDUE > 1);
	bool const vary_omega = (TORSIONS_PER_RESIDUE > 2);
}


////////////////////////////////////////////////////////////////////////////////
/// @begin small_wobble_move
///
/// @brief
///
/// @detailed
///db make  ~10 degree  changes in the torsion angles of a residue not in
///db regular secondary structure, and wobble adjacent residues to
///db reduce perturbation.  this is more conservative than a traditional
///db fragment based wobble move
///car update global position array as well as phi,psi,omega
///
/// @param  size - [in/out]? -
/// @param  cutoff_max - [in/out]? -
/// @param  nwobble - [in/out]? -
/// @param  max_wobble_gap - [in/out]? -
/// @param  total_begin - [in/out]? -
/// @param  total_end - [in/out]? -
/// @param  gfrag - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
small_wobble_move(
	int const nmoves,
	FArray1DB_int & move_list,
	int & length,
	int const nwobble,
	int const max_wobble_gap,
	int & begin,
	int & end
)
{
	int total_begin, total_end;
	small_wobble_move( nmoves, move_list, length, nwobble, max_wobble_gap, begin, end,
										 total_begin, total_end);
}

//Overloaded to permit return of total_begin and total_end to pose_small_wobble_min_trial().
void
small_wobble_move(
	int const nmoves,
	FArray1DB_int & move_list,
	int & length,
	int const nwobble,
	int const max_wobble_gap,
	int & begin,
	int & end,
	int & total_begin,
	int & total_end
)
{

	using namespace param;

//db  I tested this with nwobble=7,  wobble_gap=0, and max_wobble_gap=0
//db  I think this should be very effective for loop modeling, but logic
//db  not currently in place

	int wobble_gap;
	if ( max_wobble_gap > 0 ) {
		wobble_gap = static_cast< int >( max_wobble_gap * ran3() ) + 1;
		 // or look for close L?
	} else {
		wobble_gap = max_wobble_gap;
	}
	small_moves_wob(nmoves,move_list,length,begin,end);

	refold(begin,end);
	int dir; //refold direction
	refold_get_dir(dir);
	if ( dir == 1 ) {
		total_begin = begin;
		total_end = end + nwobble + wobble_gap;
	} else {
		total_begin = begin - ( nwobble + wobble_gap );
		total_end = end;
	}

	float rama_score, wobble_cost;
	bool gfrag( true ); // if true, powell minimized sucessfully
	if (total_begin < 1) return; //warning sign!
	add_fast_wobble(begin,end-begin+1,nwobble,wobble_gap,rama_score,wobble_cost,
	 gfrag,total_begin,total_end);
// 	std::cout << "wbl" << SS( total_begin ) << SS( total_end ) <<
// 		SS( wobble_cost ) << SS( nwobble ) << SS( begin ) << SS( end ) << std::endl;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin chuck_wobble_move
///
/// @brief
///car put in a chuck fragment, wobble it,
///car update global position array as well as phi,psi,omega
///
/// @detailed
///car 1) pick a random position and random chuck fragment passing the cutoff,
///car 2) if wobble fails, try the next frag in the list,
///car 3) if all frags in the list tried, try the adjacent position
///car 4) if no fragments found within +/- 5 of the original random position,
///car     choose another random position
///
///car wobble cutoff set to 10.0; 3-10% acceptance of moves with a wobble
///car cost of 9-10
///
///car recommended chuck cutoff (cutoff_max)=60;
///car ~65% of moves use an actual cutoff of 60/4=15
///
/// @param  size - [in/out]? -
/// @param  cutoff_max - [in/out]? -
/// @param  nwobble - [in/out]? -
/// @param  max_wobble_gap - [in/out]? -
/// @param  total_begin - [in/out]? -
/// @param  total_end - [in/out]? -
/// @param  gfrag - [in/out]? - if true, powell minimized sucessfully
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
chuck_wobble_move(
	int size,
	float cutoff_max,
	int nwobble,
	int max_wobble_gap,
	int & total_begin,
	int & total_end,
	bool & gfrag // if true, powell minimized sucessfully
)
{

	using namespace misc;
	using namespace param;
	using namespace runlevel_ns;

	float const bump_cutoff = { 5.0 }; // delta

	int const end_buffer = { 1 }; // how close to end can a frag be inserted

// local
	int size_bin;
	int wobble_gap;
	int total_buffer; // total buffer to allow (end buffer + wobble)
	int ntries_begin; // number of insertion points tried relative to
	//init_frag_begin
	int ntries_frag; // number of frags in chuck list tried
	int upper_limit,lower_limit; // absolute bounds for frag insertion
	int nfail; // total number of insertion points tried
	int c_counter,w_counter; // total number of chuck and wobble tries
	int b_counter; // number of frags bump checked
	int min_frag_begin, max_frag_begin;
	int init_frag_begin;
	int frag_begin,frag_end;
	int loop_begin,loop_end;
	int loop_num;
	bool loop_flag;
	float msd;
	float start_bump,end_bump;

	FArray1D_float cut_cost( MAX_NEIGH()() ); // cost of frags in chuck list
	FArray1D_int cut_frag( MAX_NEIGH()() ); // list of frags that pass chuck cutoff
	int cut_count; // length of chuck list
	int chuck_frag; // pointer into list of chuck fragments
	int best_frag; // pointer into neighbor list, best chuck frag
	float chuck_cost; // cost of selected chuck frag (ie best_frag)
	float chuck_cutoff; // local copy of chuck cutoff, gets set to
	// cutoff actually used
	float bump_score;
	float wobble_cutoff;
	float wobble_cost,rama_score;
	int dir; // refold direction

	FArray1D_int insert_map( MAX_RES()() );
	int total_insert;

	int const window = { 5 };
	 // how many N-terminal and C-terminal adjacent
	 // positions to try before making a new random choice

	retrieve_insertmap(insert_map,total_insert);
	if ( max_wobble_gap > 0 ) {
		wobble_gap = static_cast< int >( max_wobble_gap * ran3() ) + 1;
		 // or look for close L?
	} else {
		wobble_gap = max_wobble_gap;
	}

//car figure the limits for frag insertion
	total_buffer = end_buffer+std::max(0,nwobble+wobble_gap);
	lower_limit = insert_map(1)+total_buffer;
	upper_limit = insert_map(total_insert)-(size+total_buffer);
	min_frag_begin = lower_limit;
	max_frag_begin = upper_limit;

	loop_flag = get_loop_flag();
	w_counter = 0;
	c_counter = 0;
	b_counter = 0;
	size_bin = get_index_by_frag_size(size);
	wobble_cutoff = 10.0;
	gfrag = true;
	ntries_begin = 0;
	nfail = 0;
	init_frag_begin = 0;

L171:; // return here if chuck fails, try next insertion point

	if ( c_counter > 30 ) {  // start ramping chuck cutoff
		chuck_cutoff = float((c_counter/30)+1)*std::abs(cutoff_max);
	} else {
		chuck_cutoff = cutoff_max; // local copy,find_chuck_moves changes it...
	}
	ntries_frag = 0;
	++ntries_begin;

	if ( loop_flag ) {
		choose_random_loop(loop_begin,loop_end,loop_num);
		if ( loop_begin == 1 ) {   // insert somewhere in loop
			min_frag_begin = loop_begin + total_buffer - 1;
			max_frag_begin = loop_end - size;
		} else if ( loop_end == total_residue ) {
			min_frag_begin = loop_begin;
			max_frag_begin = loop_end - ( size + total_buffer );
		} else {                   // close loop -ignore end_buffer
			min_frag_begin = loop_begin;
			max_frag_begin = loop_end - ( size + std::max(0,nwobble+wobble_gap) );
		}

		if ( min_frag_begin > max_frag_begin ) { // not too close to ends
			++nfail;
			if ( nfail > 1000 ) {
				std::cout << "Can't find large enough loop to insert into" << std::endl;
				std::cout << "last loop :" << SS( loop_num ) << SS( loop_begin ) <<
				 SS( loop_end ) << SS( min_frag_begin ) << SS( max_frag_begin ) <<
				 SS( size ) << std::endl;
				return;
			}
			goto L171;
		}                  // check enough room to wobble
	} else {
		if ( ntries_begin == 1 ) {  // pick a random position
			min_frag_begin = lower_limit;
			max_frag_begin = upper_limit;
		} else if ( ntries_begin <= window+1 ) {  // look at next c-term position
			min_frag_begin = init_frag_begin+ntries_begin;
			max_frag_begin = min_frag_begin;
		} else if ( ntries_begin <= 2*window+1 ) { // look at next n-term position
			min_frag_begin = init_frag_begin-ntries_begin+window;
			max_frag_begin = min_frag_begin;
		}
//car double check that min and max have not moved outside acceptable range
		if ( min_frag_begin < lower_limit || max_frag_begin > upper_limit ) {
			++nfail;
			if ( nfail > 1000 ) {
				std::cout << "Too many failed trials in chuck_wobble_move" << std::endl;
				return;
			}
			goto L171;
		}
	}

	find_chuck_moves(size,chuck_cutoff,min_frag_begin,max_frag_begin,cut_cost,
	 cut_frag,cut_count,frag_begin);
	// find_chuck_moves returns actual cutoff used...

	frag_end = frag_begin + size - 1;
	++c_counter;
	if ( runlevel >= yap) std::cout << "chuck_moves" << SS( frag_begin ) <<
	 SS( min_frag_begin ) << SS( max_frag_begin ) << SS( chuck_cutoff ) <<
	 SS( cut_count ) << SS( cut_cost(1) ) << SS( ntries_begin ) <<
	 SS( c_counter ) << std::endl;

	if ( ntries_begin == 1 ) init_frag_begin = frag_begin;
	if ( ntries_begin == 11 ) ntries_begin = 0; // pick randomly next time
	if ( cut_count == 0 ) goto L171; // no frags at this position, try next

	chuck_frag = static_cast< int >( cut_count * ran3() ) + 1;
	 // pick random point in list
	chuck_cost = cut_cost(chuck_frag);
	best_frag = cut_frag(chuck_frag);

	if ( loop_flag ) {
		loop_get_best_splicemsd(loop_num,msd);
		wobble_cutoff = std::max(0.5f*chuck_cost,msd*1.2f);
		wobble_cutoff = std::max(wobble_cutoff,10.0f);
		if ( w_counter > 10 ) wobble_cutoff = std::max(wobble_cutoff,chuck_cost);
	}

L180:; // return here to try next frag in list
	if ( runlevel >= yap ) std::cout << "chuck " <<
	 ' ' << I( 3, frag_begin ) << ' ' << I( 3, best_frag ) <<
	 F( 7, 2, chuck_cost ) << F( 7, 2, chuck_cutoff ) << std::endl;
	++ntries_frag;


	static bool const replace_chuck_with_smooth = truefalseoption( "replace_chuck_with_smooth" );
	if (replace_chuck_with_smooth){ // testing alternatives to chuck
		static float const gunn_cutoff_max = 7.0;
		choose_fragment_gunn(size,gunn_cutoff_max,frag_begin);
		frag_end = frag_begin + size - 1;
	} else { //OK, do the chuck
		insert_frag(size,size_bin,frag_begin,best_frag);
	}

	refold(frag_begin,frag_begin+size-1);
	refold_get_dir(dir);
	if ( dir == 1 ) {
		total_begin = frag_begin;
		total_end = frag_end + nwobble + wobble_gap;
	} else {
		total_begin = frag_begin - ( nwobble + wobble_gap );
		total_end = frag_end;
	}
	++b_counter;
	if ( ntries_frag == 1 ) screen_move_best(total_begin,total_end,start_bump);
	screen_move_trial(total_begin,total_end,end_bump);
	bump_score = end_bump - start_bump;


	if ( runlevel >= yap) std::cout << "bump" << SS( bump_score ) <<
	 SS( end_bump ) << SS( start_bump ) << std::endl;

	if ( bump_score > bump_cutoff ) goto L300;

	if (total_begin < 1) goto L300; //warning sign!

	add_fast_wobble(frag_begin,size,nwobble,wobble_gap,rama_score,wobble_cost,
	 gfrag,total_begin,total_end);
	if ( !gfrag ) return;
	++w_counter;

	if ( runlevel >= yap ) std::cout << "wobble" << ' ' <<
	 I( 3, frag_begin ) << ' ' << I( 3, best_frag ) << ' ' <<
	 I( 3, ntries_begin ) << F( 10, 2, chuck_cost ) << F( 7, 2, chuck_cutoff ) <<
	 I( 3, ntries_frag ) << F( 10, 2, wobble_cost ) << F( 7, 1, wobble_cutoff ) <<
	 F( 7, 2, bump_score ) << std::endl;

	if ( w_counter > 20 && loop_flag ) {
		if ( runlevel >= inform) std::cout << "loop escape" << SS( w_counter ) <<
		 SS( wobble_cutoff ) << SS( wobble_cost ) << std::endl;
		goto L200; // escape for loops
	}
	if ( wobble_cost > wobble_cutoff ) goto L300;
L200:

	if ( runlevel >= yap ) std::cout << " final" << ' ' <<
	 I( 3, frag_begin ) << ' ' << I( 3, best_frag ) << ' ' << I( 3, c_counter ) <<
	 F( 7, 2, chuck_cost ) << F( 7, 2, chuck_cutoff ) << I( 3, w_counter ) <<
	 F( 7, 2, wobble_cost ) << F( 7, 1, wobble_cutoff ) << I( 4, b_counter ) <<
	 F( 6, 2, bump_score ) << F( 6, 2, bump_cutoff ) << I( 4, wobble_gap ) << std::endl;

	return;

//car jump here to abort move and decide to try either another frag at
//car the same position or a new position
L300:
	resetphipsi();
	if ( cut_count > ntries_frag ) {
		++chuck_frag;
		if ( chuck_frag > cut_count ) chuck_frag = 1;
		chuck_cost = cut_cost(chuck_frag);
		best_frag = cut_frag(chuck_frag);
		goto L180; // try another frag at same position
	} else {
		goto L171; // try a new insertion position
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin add_fast_wobble
///
/// @brief
///
/// @detailed
///car the workhorse for adding a wobble to any fragment insertion
///car frag_begin and size describe the insertion
///car only consecutive residues can be wobbled so that only the wobbled
///car residues need to be refolded on every function call
///
/// @param  frag_begin - [in/out]? -
/// @param  size - [in/out]? -
/// @param  nwobble - [in/out]? - wobble this many residues
/// @param  wobble_gap - [in/out]? - position of first wobbled residue
///        relative to frag end, -1 is first or last frag res, 0 is
///        0 is adjacent to frag, positive values are outside frag
/// @param  rama_score - [in/out]? - rama score of wobbled residues
///        (per residue)
/// @param  msd_score - [in/out]? - score calculated from change in msd
/// @param  gfrag - [in/out]? - true if minimization okay, if false, powell
///        blew up and phipsi is reset before returning
/// @param  fold_begin - [in/out]? -
/// @param  fold_end - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
add_fast_wobble(
	int & frag_begin,
	int size,
	int nwobble,
	int wobble_gap,
	float & rama_score,
	float & msd_score,
	bool & gfrag,
	int & fold_begin,
	int & fold_end,
	bool allow_use_CCD
)
{

	using namespace misc;
	using namespace param;

	//If we're in pose mode, there's actually a chance that the fragment insertion
	// occurred near a chain break. In that case, the following doesn't really make
	// sense, because the refold routines used by misc assume a continuous chain.
	// The refold routines used by pose will propagate the insertion properly... and
	// even without any wobble, the fragment insertion won't change much!!
	if (wobble_is_close_to_a_chainbreak( frag_begin, size, nwobble, wobble_gap, rama_score, msd_score, gfrag, fold_begin, fold_end )) return;

	if (get_wobble_CCD() && allow_use_CCD ){
		add_fast_wobble_CCD(frag_begin,size,nwobble,wobble_gap,rama_score,msd_score,
												gfrag,fold_begin,fold_end);
		return;
	}

	int loop_num;
	int loop_begin,loop_end;
	int msd_start,msd_res; // starting point and #res for msd calculations
	FArray2D_float pos1( 3, MAX_POS );
	FArray2D_float pos2( 3, MAX_POS );
	FArray2D_float temp_pos( 3, MAX_POS );
	float phi1,phi2;

	int dir;
	float rama,tmp1,tmp2;

	gfrag = true;

	if ( nwobble < 1 ) return;

	if ( nwobble + wobble_gap < 0 ) return; // wobble must include frag end
																					// or be outside frag
	refold(frag_begin,frag_begin+size-1); // should this be here?
																				// safe, but not efficient
	refold_get_dir(dir);

	int splint_length = nwobble;
	FArray1D_int angle_map( nwobble ); // list of residues to be minimized, local copy
	int nangles = wobble_torsions::TORSIONS_PER_RESIDUE*nwobble;
	FArray1D_float phipsi( nangles ); //packed array of angles to minimize

//car pick shorter end of chain to minimize rmsd
//cems  the number of interactions will scale with the smaller "half" on
//cems  either side of the fold. This is obviously not strictly true,
//cems  just a rule of thumb.
	int splint_begin,splint_end;

	if ( dir == 1 ) {
		splint_begin = frag_begin + size + wobble_gap;
		splint_end = splint_begin + splint_length - 1;
		msd_start = splint_end + 1; // min msd C-terminally
		msd_res = total_residue - splint_end;
		for ( int k = 1; k <= MAX_POS; ++k ) {
			for ( int j = 1; j <= 3; ++j ) {
				pos1(j,k) = Eposition(j,k,splint_begin);
				pos2(j,k) = Ebest_position(j,k,splint_end+1);
			}
		}
		phi1 = phi(splint_begin);
		phi2 = best_phi(splint_end+1);
		fold_begin = frag_begin;
		fold_end = splint_end;
	} else {
		splint_end = frag_begin - 1 - wobble_gap;
		splint_begin = splint_end - splint_length + 1;
		msd_start = 1;
		msd_res = splint_begin - 1;
		for ( int k = 1; k <= MAX_POS; ++k ) {
			for ( int j = 1; j <= 3; ++j ) {
				pos1(j,k) = Ebest_position(j,k,splint_begin);
				pos2(j,k) = Eposition(j,k,splint_end+1);
			}
		}
		phi1 = best_phi(splint_begin);
		phi2 = phi(splint_end+1);
		fold_begin = splint_begin;
		fold_end = frag_begin + size - 1;
	}

	if ( get_loop_flag() ) {
		identify_loop(frag_begin,frag_begin,loop_begin,loop_end,loop_num);
		if ( loop_begin != 1 && loop_end != total_residue ) {
			if ( loop_get_refold_dir(loop_num) == -1 ) {
				msd_start = 1;
				msd_res = loop_begin - 1;
				loop_get_best_overlap(loop_num,1,temp_pos);
				loop_halfsplice_reverse(temp_pos, /* flip fold dir for pos2 */
				 Ebest_position(1,1,loop_begin-1),pos2,1);
			} else {
				msd_start = loop_end + 1; // close the gap
				msd_res = total_residue - loop_end;
				loop_get_best_overlap(loop_num,2,temp_pos);
				loop_halfsplice_reverse(temp_pos, /* flip fold dir for pos2 */
				 Ebest_position(1,1,loop_end+1),pos2,1);
			}
		}
	}

	if ( msd_res < 1 ) {    // no residues for msd --> chain end
		msd_score = -1.0;
		rama_score = 0.0;
		for ( int i = splint_begin; i <= splint_end; ++i ) {
			eval_rama_score_residue(res(i),phi(i),psi(i),secstruct(i),rama,tmp1,tmp2);
			rama_score += rama;
		}
		goto L100;
	}

	splint_precompute(pos1,phi1,pos2,phi2,Ebest_position(1,1,msd_start),msd_res);


// make a list of the residue numbers in the splint that which will be
// minimized and store in namespace to be unpacked in minimizer
	for ( int i = 1; i <= splint_length; ++i ) {
		angle_map(i) = splint_begin + i - 1;
		name(splint_begin + i - 1) = "-WB-";
	}

//pack up phipsi array for minimizer; generate splint arrays
	wobble_pack_phipsi(angle_map,splint_length,phipsi,nangles);
	build_splint(splint_begin,splint_end,angle_map,splint_length);
	gfrag = find_splint(phipsi,nangles,msd_score,rama_score);
	if ( ! gfrag ) return;
//car put splint phispi into global arrays
	wobble_unpack_phipsi(phipsi,angle_map,splint_length,phi,psi,omega);

L100:
	refold(fold_begin,fold_end);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin add_wobble
///
/// @brief
///car this function can handle wobbling non-consecutive residues,
///car but does so at the expense of refolding the entire frag
///car insertion plus wobble region
///
/// @detailed
///
///car the workhorse for adding a wobble to any fragment insertion
///car frag_begin and size are describe the insertion
///
/// @param  frag_begin - [in/out]? -
/// @param  size - [in/out]? -
/// @param  mres_gap1 - [in/out]? - position of first wobbled residue at N-term
///                    relative to frag end, -1 is first frag res, 0 is residue
///                    preceding frag, positive values are outside frag
/// @param  mres_gap2 - [in/out]? - position of first wobbled residue at c-term
///                    relative to frag end, -1 is last frag res, 0 is residue
///                    following frag, positive values are outside frag
/// @param  mres_before - [in/out]? - wobble this many residues at N-term end of frag
/// @param  mres_after - [in/out]? - wobble this many residues at C-term end of frag
/// @param  rama_score - [in/out]? - score calculated from change in msd
/// @param  msd_score - [in/out]? - score calculated from change in msd
/// @param  gfrag - [in/out]? - true if okay, false if powell blew up
///                and phipsi is reset before returning
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
add_wobble(
	int & frag_begin,
	int size,
	int & mres_gap1,
	int & mres_gap2,
	int & mres_before,
	int & mres_after,
	float & rama_score,
	float & msd_score,
	bool & gfrag // true if okay, false if powell blew up
)
{

	using namespace misc;
	using namespace param;

	int loop_num;
	int loop_begin,loop_end;

	int msd_start,msd_res; // starting point and #res for msd calculations
	FArray2D_float pos1( 3, MAX_POS );
	FArray2D_float pos2( 3, MAX_POS );
	FArray2D_float temp_pos( 3, MAX_POS );


	int wobble_begin = frag_begin-mres_gap1-mres_before; // first wobbled res
	int wobble_end = frag_begin+size-1+mres_gap2+mres_after; // last wobbled res
	if ( wobble_end < wobble_begin ) return;

	int splint_begin = std::min(frag_begin,wobble_begin); // first in splint
	int splint_end = std::max(wobble_end,frag_begin+size-1); // last in splint

	for ( int k = 1; k <= MAX_POS; ++k ) {
		for ( int j = 1; j <= 3; ++j ) {
			pos1(j,k) = Ebest_position(j,k,splint_begin);
			pos2(j,k) = Ebest_position(j,k,splint_end+1);
		}
	}

//car pick shorter end of chain to minimize rmsd
//cems  the number of interactions will scale with the smaller "half" on
//cems  either side of the fold. This is obviously not strictly true,
//cems  just a rule of thumb.
	if ( total_residue-splint_end < splint_begin-1 ) {
		msd_start = splint_end+1; // min msd C-terminally
		msd_res = total_residue-splint_end;
	} else {
		msd_start = 1; // min msd N-terminally
		msd_res = splint_begin - 1;
	}

	if ( get_loop_flag() ) {
		identify_loop(frag_begin,frag_begin,loop_begin,loop_end,loop_num);
		if ( loop_begin != 1 && loop_end != total_residue ) {
			if ( loop_get_refold_dir(loop_num) != 1 ) {
				std::cout << "N2C folding required for wobble with loops" << std::endl;
				return;
			}
			msd_start = loop_end+1; // close the gap
			msd_res = total_residue-loop_end;
			loop_get_best_overlap(loop_num,2,temp_pos);
			loop_halfsplice_reverse(temp_pos, /* flip fold dir for pos2 */
			 Ebest_position(1,1,loop_end+1),pos2,1);
		}
	}

	splint_precompute(pos1,best_phi(splint_begin),pos2,best_phi(splint_end+1),
								 Ebest_position(1,1,msd_start),msd_res);

// make a list of the residue numbers in the splint that which will be
// minimized; phipsi is used to pass these to the minimizer

//car length of map, use min so we don't overcount
//car note this works cause of order in which residues placed in map
//car counting forward from wobble begin, then backwards from wobble_end
//car so if there is overlap, duplicated residues are at the end of
//car the array and will be ignored
	int mres_count = std::min(mres_before+mres_after,wobble_end-wobble_begin+1);

//car dimension angle_map to larger size because we will write to
//car the entire array even though we will not pass it to the minimizer
	FArray1D_int angle_map( mres_before + mres_after ); // list of residues to be minimized, local copy
	for ( int i = 1; i <= mres_before; ++i ) {
		angle_map(i) = wobble_begin + i - 1;
		name( wobble_begin + i - 1 ) = "-WM-"; // dummy name marks begin wobble
	}
	for ( int i = 1; i <= mres_after; ++i ) {
		angle_map(i+mres_before) = wobble_end + 1 - i;
		name( wobble_end + 1 - i )  = "-AV-"; // marks end wobble
	}

//pack up phipsi array for minimizer
	int nangles = wobble_torsions::TORSIONS_PER_RESIDUE*mres_count;
	FArray1D_float phipsi( nangles ); //packed array of angles to minimize
	wobble_pack_phipsi(angle_map,mres_count,phipsi,nangles);
	build_splint(splint_begin,splint_end,angle_map,mres_count);
	gfrag = find_splint(phipsi,nangles,msd_score,rama_score);
	if ( ! gfrag ) return;
//car put splint phispi into global arrays
	wobble_unpack_phipsi(phipsi,angle_map,mres_count,phi,psi,omega);
	refold(splint_begin,splint_end);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin splint_set_com_weight
///
/// @brief
///
/// @detailed
/// this option allows you to specify how much to weight the COM relative
/// to the translation in computing the RMSD change.
/// If this value is one then the rmsd change
/// will be literally the rmsd change.  if this value is zero then the rmsd
/// computed will be the due to only rotation effect and pistoning and hurling
/// motion of COM will be ignored. In between you get what you ask for.
///
/// @param  com_weight_x - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
splint_set_com_weight( float com_weight_x )
{

	using namespace splint_piston;

	com_weight = com_weight_x;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin splint_get_com_weight
///
/// @brief
///
/// @detailed
///
/// @param  com_weight_x - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
splint_get_com_weight( float & com_weight_x )
{
	using namespace splint_piston;

	com_weight_x = com_weight;
}
////////////////////////////////////////////////////////////////////////////////
/// @begin build_splint
///
/// @brief
///   create splint arrays for folding and function/deriv calculations
///
/// @detailed
///
/// @param  xbegin - [in] - begining of splint (region to fold)
/// @param  xend - [in] - end of splint
/// @param  xmap - [in] - list of residues to wobble (minimize)
/// @param  xcount - [in] - logical length of xmap
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors  car 9/12/2004
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
build_splint(
	int const xbegin,
	int const xend,
	FArray1Da_int xmap,
	int const xcount
)
{
	using namespace wobble_splint;
	using namespace misc;


//prepare splint arrays for folding
	splint_length = xend - xbegin + 1;
	if ( splint_length > MAX_SPLINT ) {
		std::cout << "splint length " << splint_length << " greater than MAX_SPLINT" <<
			MAX_SPLINT << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// copy global phi, psi,omega into splint phi,psi,omega
	for ( int i = xbegin, j = 1; i <= xend; ++i, ++j ) {
		splint_phi(j) = phi(i);
		splint_psi(j) = psi(i);
		splint_omega(j) = omega(i);
		splint_ss(j) = secstruct(i);
		splint_res(j) = res(i);
	}

	for ( int i = 1; i <= xcount; ++i ) {
		splint_angle_map(i) = xmap(i) - xbegin + 1;
	}
	splint_map_length = xcount;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin splint_precompute
///
/// @brief
/// compute the ellisoid arrays needed to rapidly apply rotations.
/// these are stored as a package global.
///
/// @detailed
/// calculates rotation and offset to move pos2  onto pos1.
/// calculates elliptic cross moments between ellipse at pos3 and
/// ellipse translated and rotated from pos2 to pos1
///
/// @param  pos1 - [in] - splint_begin coords
/// @param  pos2 - [in] - splint_end coords- pos1/pos2 describes move to
///               unmake in best position that will be made by the
///               splint + frag insertion
/// @param  phi1 - [in] -
/// @param  phi2 - [in] -
/// @param  pos3 - [in] - current location of atoms for msd (ptr into full array)
/// @param  nres - [in] - number of res over which to calc msd
///               (ie nres in Ebest array)
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
splint_precompute(
	FArray2Da_float pos1,
	float const phi1,
	FArray2Da_float pos2,
	float const phi2,
	FArray3Da_float pos3,
	int const nres
)
{
	using namespace cold;
	using namespace old_transform;
	using namespace param;
	using namespace splint_elipse;

	if ( nres < 1 ) {
		std::cout << "ERROR in splint_precompute!!" << std::endl;
		std::cout << "no atoms for eliptic_cross_moments" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	pos1.dimension( 3, MAX_POS );
	pos2.dimension( 3, MAX_POS );
	pos3.dimension( 3, MAX_POS, nres );

	FArray2D_float ZtC( 3, 3 );
	FArray1D_float O2( 3 );
	compute_frag_move(pos1,phi1,pos2,phi2,A2,U2,ZtC,O2);
// 	std::cout << "ANGLES" << SS(phi1) << SS(phi2) << std::endl;
// 	debug_mat("pos1*",pos1);
// 	debug_mat("pos2*",pos2);
// 	debug_mat("pos3*",pos3);

	int const natoms = nres * MAX_POS;
	eliptic_cross_moments(pos3,natoms,atom_weights,A2,U2,O2,C_EL,EL2,EL,ZtC);
// 	std::cout << "splint cross moments" << std::endl;
// 	debug_mat("EL*",EL);
// 	debug_mat("EL2*",EL2);
// 	debug_mat("C_EL*",C_EL);
// 	debug_3("U2 O2 O2*",U2,O2,O2);
// 	debug_vec("A2*",A2);
// 	debug_mat("ZtC*",ZtC);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin splint_compute
///
/// @brief
///
/// @detailed
///
/// @param  el_score - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
splint_compute()
{
	using namespace splint_elipse;

// local
	FArray2D_float A2( 3, 3 );
	FArray1D_float ni( 3 );
	FArray1D_float cc( 3 );
	FArray1D_float ca( 3 );
	float rcm,com_weight,el_score;

	angles_get_final_stub(ni,ca,cc);
// 	std::cout << "splint final stub" << std::endl;
// 	debug_vec("ni**",ni);
// 	debug_vec("ca**",ca);
// 	debug_vec("cc**",cc);

	angles_coord_sys(ni,ca,cc,A2);
	eliptic_rmsd_2(A2,ca,EL,EL2,C_EL,1.0e9f,el_score,rcm);


//  	std::cout << "splint cost" << std::endl;
//  	debug_mat("A2*",A2);
//  	debug_vec("ca*",ca);

	splint_get_com_weight(com_weight);
	el_score += (com_weight-1)*rcm;
	return el_score;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin find_splint
///
/// @brief
///
/// @detailed
///
/// @param  phipsi - [in] -  angles to minimize
/// @param  nangles - [in] -  logical length of phipsi
/// @param  msd_score - [out] -
/// @param  rama_score - [out] -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
find_splint( FArray1Da_float phipsi, int nangles, float & msd_score, float & rama_score)
{
	phipsi.dimension( nangles );

	int iter;
	float fret;
	bool gfrag( true );

	minimize_set_func(1);   //pass calls to func to func_wobble

// initial score
//	fret = func_wobble(phipsi, gfrag, msd_score, rama_score);
//	std::cout << "wob chuck cost" << SS( msd_score ) << std::endl;

//car  db reports that for 4 wobbled residues, ~65% of moves are minimized
//car  to func ~6
//car  for only 1-2 residues, powell performs better than dfpmin, presumably
//car  because rama score is not in the derivative calculation
	if ( nangles > 4 ) {
		dfpmin(phipsi,nangles,0.001,iter,fret,gfrag);
	} else {
		powell(gfrag,phipsi,nangles,0.1,iter,fret);
	}
// get final score for return
	fret = func_wobble(phipsi, gfrag, msd_score, rama_score);

	return gfrag;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin func_wobble
///
/// @brief
///
/// @detailed
///
/// @param  phipsi - [in/out]? -
/// @param  gfrag - [in/out]? -
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
func_wobble(
	FArray1Da_float phipsi,
	bool & gfrag,
	float & msd_score,
	float & rama_score
)
{

	using namespace param;


	gfrag = splint_refold(phipsi);
	if (! gfrag ) {
		rama_score = 0.0;
		msd_score = 0.0;
		return 0.0;  //return bogus value;
	}
	wobble_cost(rama_score, msd_score);

//cj original
//cj func = msd_score
//cj 24Mar00 want to jump between islands so not log
//cj msd_score will vary up to 20 (around 2 is accepted, 10-20 is bad)
//cj bump_score is ramachandran score and will be 2 good, 6 okay, & 14 bad
//cj 06Apr00 want penalty if above zero, so square otherwise use straight

	return (std::abs(msd_score)+0.04f) + rama_score;
//car Increase weight to 2-3?? why .04?
}

////////////////////////////////////////////////////////////////////////////////
/// @begin splint_refold
///
/// @brief
///  fold splint so that stub is calculated; toss the generated coordinates
///
/// @detailed
///
/// @param  phipsi - [in] -
/// @param  gfrag - [out] -
///
/// @return  reference to splint coordinates
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

bool
splint_refold(FArray1Da_float phipsi)
{
	using namespace param;
	using namespace param_aa;
	using namespace wobble_splint;

	if ( wobble_unpack_phipsi(phipsi,splint_angle_map,splint_map_length,
		 splint_phi,splint_psi,splint_omega) ) {

//car variables for old refold, unused and uninitialized
		static FArray3D_float bpos( 3, MAX_POS, MAX_SPLINT );
		static FArray2D_float bcen( 3, MAX_SPLINT );
		static FArray2D_float cen( 3, MAX_SPLINT );
		static FArray3D_float bfull( 3, MAX_ATOM(), MAX_SPLINT );
		static FArray3D_float full( 3, MAX_ATOM(), MAX_SPLINT );
		static FArray1D_int res( MAX_SPLINT, aa_ala ); // set arbitrarily to ala
		static FArray1D_int resv( MAX_SPLINT, 1 );     // set arbitrarily to 1

		backbone_fold(splint_length, splint_phi, splint_psi, splint_omega, splint_xyz);
		return true;
	}

	return false;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin wobble_unpack_phipsi
///
/// @brief copies phipsi angles into the global
///
/// @detailed
///
/// @param  phipsi - [in] -
/// @param  phi_out - [out] -
/// @param  psi_out - [out] -
/// @param  omega_out - [out] -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool
wobble_unpack_phipsi(
	FArray1Da_float phipsi,
	FArray1Da_int angle_map,
	int const map_length,
	FArray1Da_float phi_out,
	FArray1Da_float psi_out,
	FArray1Da_float omega_out
)
{
	using namespace wobble_torsions;

	int nangles =  map_length*TORSIONS_PER_RESIDUE;
	phipsi.dimension( nangles );

//car check angles in range
	for ( int i = 1; i <= nangles; ++i ) {
		if ( std::abs(phipsi(i)) > 360.0 ) {
			angle_in_range(phipsi(i));
			assert( phipsi(i) <= 180.0 && phipsi(i) > -180.0 );
		}
	}

// copy phipsi into output arrays
	for ( int i = 1; i <= map_length; ++i ) {
		if ( phipsi(i) != phipsi(i) ) {
		 // NaN detected (works for IEEE floating point)
			std::cout << "WOBBLE ERROR" << std::endl;
			std::cout << "angle_type:   phi" << std::endl;
			std::cout << "residue:    " << angle_map(i) << std::endl;
			std::cout << "value:      " << phipsi(i) << std::endl;
			return false;
		}
		phi_out(angle_map(i)) = phipsi(i);

		if (vary_psi) {
			int j = i + map_length;
			if ( phipsi(j) != phipsi(j) ) {
				// NaN detected (works for IEEE floating point)
				std::cout << "WOBBLE ERROR" << std::endl;
				std::cout << "angle_type:   psi" << std::endl;
				std::cout << "residue:    " << angle_map(i) << std::endl;
				std::cout << "value:      " << phipsi(j) << std::endl;
				return false;
			}
			psi_out(angle_map(i)) = phipsi(j);
		}

		if  (vary_omega) {
			int j = i + map_length * 2;
			if ( phipsi(j) != phipsi(j) ) {
				// NaN detected (works for IEEE floating point)
				std::cout << "WOBBLE ERROR" << std::endl;
				std::cout << "angle_type:   omega" << std::endl;
				std::cout << "residue:    " << angle_map(i) << std::endl;
				std::cout << "value:      " << phipsi(j) << std::endl;
				return false;
			}
			omega_out(angle_map(i)) = phipsi(j);
		}
	}
	return true;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin wobble_pack_phipsi
///
/// @brief
///
/// @detailed
///
/// @param  xmap - [in] -  list of residues to torsion-optimize
/// @param  xcount - [in] -  length of residue list
/// @param  xbegin - [in] - first residue to refold
/// @param  xend - [in] -  last residue to refold
/// @param  phipsi - [out] -  packed array of angles to minimize
///
/// @global_read
///
/// @global_write
///
/// @remarks
///   assumes that angles are in order phi-for-all-residues, psi-for-all-residues
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
wobble_pack_phipsi(
	FArray1Da_int xmap,
	int const xcount,
	FArray1Da_float phipsi,
	int const nangles
)
{
	using namespace wobble_torsions;
	using namespace misc;

	assert(TORSIONS_PER_RESIDUE < 4);

	xmap.dimension( xcount );

	if ( nangles < xcount*TORSIONS_PER_RESIDUE) {
		std::cout << " STOPPING:: Declared size of phipsi " << nangles <<
		 " insufficient to contain " << xcount << " residues with " <<
		 TORSIONS_PER_RESIDUE << " angles per residue " << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	for ( int i = 1; i <= xcount; ++i ) {
		int res = xmap(i);
		if (res < 1 || res > total_residue) {
			std::cout << "invalid residue number in wobble " << res << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		phipsi(i)            = phi(res);
		if ( vary_psi ) phipsi(i+xcount) = psi(res);
		if ( vary_omega ) phipsi(i + 2*xcount) = omega(res);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin wobble_cost
///
/// @brief
///
/// @detailed
///
/// @param  rama_score - [in/out]? -
/// @param  msd_score - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
wobble_cost(
	float & rama_score,
	float & msd_score
)
{
	using namespace wobble_splint;


	float score, dphi, dpsi;

	rama_score = 0.0;

	for ( int i = 1; i <= splint_map_length; ++i ) {
		int j = splint_angle_map(i);

		eval_rama_score_residue(splint_res(j),splint_phi(j),splint_psi(j),
			splint_ss(j), score, dphi, dpsi);
		rama_score += score;
	}
	rama_score /= splint_map_length;   // normalize by map_length?

// assumes structure has been refolded already
	msd_score = splint_compute();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin angle_in_range
///
/// @brief
///car returns the equivalent angle in degrees in the range (-180.0 < ang <= 180.0)
///
/// @detailed
///
/// @param  ang - [in/out] -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
angle_in_range( float & ang )
{
	int const odd = std::abs( mod( static_cast< int >( ang / 180.0f ), 2 ) ); // 1 if ang/180 is odd,0 if ang/180 is even
	ang = mod( ang, 180.0f );
	if ( odd == 0 ) return;
	if ( ang > 0.0 ) {
		ang -= 180.0f;
		return;
	}
	ang += 180.0f;
	assert( ang <= 180.0f && ang > -180.0f);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin dfunc_wobble
///
/// @brief
///
/// @detailed
///db dfunc_wobble computes the derivative of f=(V-V')**2 with respect to each
///db of the torsion angles in the splint.  Using the chain rule, we have
///db
///db df/d phi = df/dri dri/dphi = 2ri dri/dphi.
///db
///db dri/dphi = Eab x (V-Vb) * (V' - V)/|V-V'|
///db
///db (the first cross produce is the displacement of V upon a rotation dphi
///db around the unit vector Eab, Vb is the coordinates of the second atom in
///db the bond)
///db
///db since | V-V'| = ri,
///db
///db df/ dphi = 2 Eab x (V-Vb) * (V' - V) =
///db
///db  note that Eab and Vb are different for each torsion angle, but V'
///db and V are the same.  rearranging:
///db
///db = -  2 Eab X Vb * (V' - V) - 2 Eab * (V' x V).
///db
///db now we need the averages over all Vi of the difference and the
///db crossproduct of V and V'.
///db
///
/// @param  phipsi - [in/out]? -
/// @param  dE_dphipsi - [in/out]? -
/// @param  n - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
dfunc_wobble(
	FArray1Da_float phipsi,
	FArray1Da_float dE_dphipsi,
	int n
)
{
	using namespace wobble_splint;
	using namespace old_transform;
	using namespace splint_elipse;
	using numeric::conversions::radians;

	phipsi.dimension( n );
	dE_dphipsi.dimension( n );

//car see also comments in elliptic_msd.cc
//car V is an arbitrary vector

//car local
	int end_residue(0),end_atom(0);
	FArray1D_float ni( 3 );
	FArray1D_float cc( 3 );
	FArray1D_float vcross( 3 );
	FArray1D_float vdiff( 3 );
	FArray1D_float ExB( 3 );
	FArray1D_float vec( 3 );
	FArray2D_float R( 3, 3 );
	FArray1D_float L( 3 );
	FArray1D_float T( 3 );
	FArray1D_float temp( 3 );
	FArray1D_float ELd( 10 );
	float com_weight;

	FArray2D_float A1( 3, 3 );
	FArray1D_float U1( 3 );

//car variables for numerical deriv check
//$$$      FArray1D_float phipsi_ori(4);
//$$$      float f1,f2;
//$$$      int j;
//$$$      float func,deriv;

//db     the following lines compute the derivative numerically as a check
//       gfrag = true;
//       for ( i = 1; i <= n; ++i ) {
//         phipsi_ori(i) = phipsi(i);
//       }

//       for ( i = 1; i <= n; ++i ) {
//         for ( j = 1; j <= 4; ++j ) {
//           phipsi_ori(i) = phipsi(i)+float(j)*.5;
//           f1 = func(gfrag,phipsi_ori);
//           phipsi_ori(i) = phipsi(i)-float(j)*.5;
//           f2 = func(gfrag,phipsi_ori);
//           deriv = (f1-f2)/float(j);
//           phipsi_ori(i) = phipsi(i);
//           std::cout << SS( i ) << SS( j ) << SS( deriv ) << std::endl;
//         }
//       }

	splint_get_com_weight(com_weight);

	bool gfrag = splint_refold(phipsi);
	if ( !gfrag ) return; // set deriv to 0?

// get rotation matrix corresponding to current splint
	angles_get_final_stub(ni,U1,cc);
	angles_coord_sys(ni,U1,cc,A1); // A1^transpose * X + U1 ismovementofthis frag
	mat_multiply_transpose3(A1,A2,R); // R = A1^transpose * A2
	rotate(R,U2,T);
	subvec(U1,T,L); // L = U1 - R U2

	rotate(R,EL(7),temp);

	for ( int i = 1; i <= 3; ++i ) {
		float const EL_6i = EL(6+i);
		ELd(i) = EL_6i * EL_6i;
	}

	ELd(4) = EL(7)*EL(8);
	ELd(5) = EL(7)*EL(9);
	ELd(6) = EL(8)*EL(9);

//db  first compute V x V' = V x ( R V + L)

//db   vcross(1) = V(y)V'(z) - V(z)V'(y)
	vcross(1) = R(1,3)*ELd(4) + R(2,3)*ELd(2) + R(3,3)*ELd(6);
	vcross(1) += -R(1,2)*ELd(5) - R(2,2)*ELd(6) - R(3,2)*ELd(3);
	vcross(1) += L(3)*EL(8) - L(2)*EL(9);
	vcross(2) = R(1,1)*ELd(5) + R(2,1)*ELd(6) + R(3,1)*ELd(3);
	vcross(2) += -R(1,3)*ELd(1) - R(2,3)*ELd(4) - R(3,3)*ELd(5);
	vcross(2) += L(1)*EL(9) - L(3)*EL(7);
	vcross(3) = R(1,2)*ELd(1) + R(2,2)*ELd(4) + R(3,2)*ELd(5);
	vcross(3) += -R(1,1)*ELd(4) - R(2,1)*ELd(2) - R(3,1)*ELd(6);
	vcross(3) += L(2)*EL(7) - L(1) * EL(8);

	for ( int i = 1; i <= 3; ++i ) {
		vcross(i) *= com_weight;
	}

	vcross(1) += R(1,3) * EL(4) + R(2,3) * EL(2) + R(3,3) * EL(6);
	vcross(1) += - R(1,2) * EL(5) - R(2,2) * EL(6) - R(3,2) * EL(3);
	vcross(2) += R(1,1) * EL(5) + R(2,1) * EL(6) + R(3,1) * EL(3);
	vcross(2) += - R(1,3) * EL(1) - R(2,3) * EL(4) - R(3,3) * EL(5);
	vcross(3) += R(1,2) * EL(1) + R(2,2) * EL(4) + R(3,2) * EL(5);
	vcross(3) += - R(1,1) * EL(4) - R(2,1) * EL(2) - R(3,1) * EL(6);

//db   vdiff = (V' - V)

	vdiff(1) = (R(1,1)-1.) * EL(7) + R(2,1) * EL(8) + R(3,1) * EL(9) + L(1);
	vdiff(2) = R(1,2) * EL(7) + (R(2,2)-1.) * EL(8) + R(3,2) * EL(9) + L(2);
	vdiff(3) = R(1,3) * EL(7) + R(2,3) * EL(8) + (R(3,3)-1.) * EL(9) + L(3);

	for ( int i = 1; i <= 3; ++i ) {
		vdiff(i) *= com_weight;
	}

//db     for each torsion, deriv = ( - vcross * E + vdiff * (E X B) )
//db     where E is unit vector along torsion and B is vector to second
//db     point in bond

	int nwobble = n/wobble_torsions::TORSIONS_PER_RESIDUE;
	for ( int j = 1; j <= nwobble; ++j ) {
		int i = splint_angle_map(j);

		for ( int phi_or_psi = 1; phi_or_psi <= wobble_torsions::TORSIONS_PER_RESIDUE; ++phi_or_psi ) {
			make_unit_vector_from_splint(i,phi_or_psi,vec,end_atom,end_residue);
			cros(vec,splint_xyz(1,end_atom,end_residue),ExB);

			dE_dphipsi(i+nwobble*(phi_or_psi-1)) =
			 radians( dotprod(vcross,vec) - dotprod(vdiff,ExB) ) * 2.0f;
// 			std::cout << SS( i ) << SS(phi_or_psi) << SS(nwobble) <<
// 			 SS( dE_dphipsi(i+nwobble*(phi_or_psi-1)) ) <<
// 			 SS( -dotprod(vcross,vec) ) << SS( dotprod(vdiff,ExB) ) << std::endl;


//db the following computes the derivative for the straightforward single
//db  point case
//      if ( com_weight > 20. ) {
//        subvec(temp,Eposition(1,end_atom,end_residue),uxr);
//        cros(vec,uxr,Exr);
//        dE_dphipsi(i+splint_length*(phi_or_psi-1)) = radians( dotprod(ExR,vdiff) ) * 2.
//        std::cout << "newderiv" << SS( dE_dphipsi(i+splint_length*(phi_or_psi-1)) ) << std::endl;
//      }

		}
	}

//db  i didn't try resurrecting the following. with the deriv correct,
//db  there doesnt seem to be any need for it.

// now fix a problem:  step size in MNBRAK is too big given the periodicity of the angular sur
// keeping same direction for linmin.  as a kludge we restrict the max RMSDof the derivative
// will now attempt to restore some sanity to the step size.  this may not be thebest way to
// we compute the length/sqrt(ndim) of the derivative vector and if it's larger than say 10, r
// its length to  10*sqrt(Ndim).  this scalling means the rms size of any one dev value is 1

//      norm = 0.0;
//      for ( i = 1; i <= map_length; ++i ) {
//         norm += square( dE_dphipsi(i) ) + square( dE_dphipsi(i+map_length) );
//      }
//      norm = std::sqrt(norm/(2.0*map_length));
//      if ( norm > 10 ) {
//         norm = 10.0/norm;
//         for ( i = 1, e = map_length * 2; i <= e; ++i ) {
//            dE_dphipsi(i) *= norm;
//         }
//      }

}

////////////////////////////////////////////////////////////////////////////////
/// @begin make_unit_vector_from_splint
///
/// @brief
///car returns the unit vector along phi,psi,omega of residue i
///car also returns the atom and residue index  C-terminal atom in the bond
///
/// @detailed
///
/// @param  i - [in/out]? -
/// @param  phi_or_psi - [in/out]? -
/// @param  vec - [in/out]? -
/// @param  end_atom - [in/out]? -
/// @param  end_res - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
make_unit_vector_from_splint(
	int & i,
	int & phi_or_psi,
	FArray1Da_float vec,
	int & end_atom,
	int & end_res
)
{
	using namespace wobble_splint;

	vec.dimension( 3 );

	int start_atom = 0;

	if ( phi_or_psi == 1 ) {     // phi
		start_atom = 1;
		end_atom = 2;
		end_res = i;
	} else if ( phi_or_psi == 2 ) {   // psi
		start_atom = 2;
//db     WARNING  this is for the position array!! (not the full_coord array)
		end_atom = 4;
		end_res = i;
	} else if ( phi_or_psi == 3 ) { // omega
		start_atom = 3;
		end_atom = 1;
		end_res = i+1;
	} else {
		std::cout << "Invalid value of phi_or_psi: " << phi_or_psi << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	vec(1) = splint_xyz(1,end_atom,end_res) - splint_xyz(1,start_atom,i);
	vec(2) = splint_xyz(2,end_atom,end_res) - splint_xyz(2,start_atom,i);
	vec(3) = splint_xyz(3,end_atom,end_res) - splint_xyz(3,start_atom,i);

	vec.normalize();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin screen_move_trial
///
/// @brief
///car compute bumps between a fragment in the position array and the
///car rest of the best_position array; gives the estimated contribution
///car to the vdw_score of a fragment insertion
///
/// @detailed
///
/// @param  insert_start - [in/out]? -
/// @param  insert_end - [in/out]? -
/// @param  bump_score - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
screen_move_trial(
	int & insert_start,
	int & insert_end,
	float & bump_score
)
{

	using namespace misc;

	screen_move(insert_start,insert_end,total_residue,position,best_position,
	 centroid,best_centroid,atom_type,atom_type_cen,bump_score);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin screen_move_best
///
/// @brief
///car compute bumps between a fragment in the best_position array and the
///car rest of the best_position array; gives the estimated contribution
///car to the vdw_score of the current 'fragment' in the structure.
///
/// @detailed
///
/// @param  insert_start - [in/out]? -
/// @param  insert_end - [in/out]? -
/// @param  bump_score - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
screen_move_best(
	int & insert_start,
	int & insert_end,
	float & bump_score
)
{

	using namespace misc;

	screen_move(insert_start,insert_end,total_residue,best_position,best_position,
	 best_centroid,best_centroid,atom_type,atom_type_cen,bump_score);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin screen_move
///
/// @brief
///
/// @detailed
///car estimates the change in vdw_score that will result from a
///car fragment insertion when some method will be used to minimize
///car long range effects of the insertion (ie double or wobble move)
///car if the downstream chain completely returns to its original position,
///car then the increase in vdw_score will result only from bumps between
///car the 'insert' region and the fixed region (ie the region outside
///car the two insert).
///
///car calculates the bump score (analagous to vdw_score) between
///car 1. the position array in the insert region and  2. the _BEST_position
///car array for the region outside the insertion
///
///car it is the caller's responsibility to generate the position array
///car such that it reflects the possible bumps.
///car generally, want to fold in the direction that maximizes lever effects
///car (ie frag1 N-term to frag2 or wobble, fold N->C  dir=1
///car     frag2 C-term to frag2 or wobble, folc C->N  dir=1
///car     insert_begin,insert_edn should include entire frag1 to frag2 region)
///
///car based on vdw.cc with the internal arrays stripped out
///  1) if the centroids >  12 A apart, no atoms to residues can bump
///  2) if CA(i) CA(j) distance > 7A, no backbone atoms of pair bump
///
/// @param  insert_start - [in/out]? -
/// @param  insert_end - [in/out]? -
/// @param  total_residue - [in/out]? -
/// @param  position - [in/out]? -
/// @param  best_position - [in/out]? -
/// @param  centroid - [in/out]? -
/// @param  best_centroid - [in/out]? -
/// @param  atom_type - [in/out]? -
/// @param  atom_type_cen - [in/out]? -
/// @param  bump_score - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
screen_move(
	int & insert_start,
	int & insert_end,
	int & total_residue,
	FArray2Da_float position,
	FArray2Da_float best_position,
	FArray2Da_float centroid,
	FArray2Da_float best_centroid,
	FArray1Da_short atom_type,
	FArray1Da_short atom_type_cen,
	float & bump_score
)
{
	using namespace cenlist_ns;
	using namespace param;

	position.dimension( 3, MAX_POS * MAX_RES()() );
	best_position.dimension( 3, MAX_POS * MAX_RES()() );
	centroid.dimension( 3, MAX_RES() );
	best_centroid.dimension( 3, MAX_RES() );
	atom_type.dimension( MAX_POS * MAX_RES()() );
	atom_type_cen.dimension( MAX_RES() );

//car local
	int type0; // centroid i
	int type1; // atom j
	int type2; // atom i
	int type3; // centroid j
	double xpos1,ypos1,zpos1; // coord of j atom
	double xcen0,ycen0,zcen0; // coord of i centroid
	double xcen3,ycen3,zcen3; // coord of j centroid
	double tempdist,dist2,alpha_alpha;

	bump_score = 0.0;
	for ( int i = insert_start; i <= insert_end; ++i ) {
		int const itemp = MAX_POS*(i-1)+1; // pentamer starting point
		type0 = atom_type_cen(i);

		xcen0 = centroid(1,i); // prelook up frequently used
		ycen0 = centroid(2,i);
		zcen0 = centroid(3,i);

		for ( int j = 1; j <= total_residue; ++j ) {
			int const jtemp = MAX_POS*(j-1)+1; // pentamer starting point
			if ( j >= insert_start && j <= insert_end ) goto L100; // next j
			xcen3 = best_centroid(1,j); // prelook up frequently used
			ycen3 = best_centroid(2,j);
			zcen3 = best_centroid(3,j);

//---------
// centroid-centroid
//---------
			{ // Scope
				double const xdif = xcen0 - xcen3;
				double const ydif = ycen0 - ycen3;
				double const zdif = zcen0 - zcen3;
				tempdist = ( xdif * xdif ) + ( ydif * ydif ) + ( zdif * zdif );
			}

			if ( tempdist > cen_dist_cutoff2 ) goto L100; // next j
			type3 = atom_type_cen(j); // use this later twice
			dist2 = vdw::atom_vdw(type0,type3) - tempdist;

			if ( dist2 > 0.0 ) bump_score += dist2; // clash score

// check all atoms
			if ( j-i <= 1 ) goto L100; // skip adjacent residues; next j


// as a prefilter we first compute the calpha-calpha distance.  if this is
// greater than 49 (7 squared) { none of the backbone/cbeta can clash.
// and we skip the distance backbone checks.
// additionally if this is greater than 60 then
// we can skip the centriod backbone checks.

			{ // Scope
				double const dif1 = position(1,itemp+1) - best_position(1,jtemp+1);
				double const dif2 = position(2,itemp+1) - best_position(2,jtemp+1);
				double const dif3 = position(3,itemp+1) - best_position(3,jtemp+1);
				 // plus one means calpha
				alpha_alpha = ( dif1 * dif1 ) + ( dif2 * dif2 ) + ( dif3 * dif3 );
			}

			for ( int k = jtemp; k <= jtemp+4; ++k ) { // all atoms in j res
				type1 = atom_type(k);
				if ( type1 != 0 ) { // Not gly c-beta
//---------
// i-centroid to j-pentamer
//---------
					double const xdif = xcen0 - best_position(1,k);
					double const ydif = ycen0 - best_position(2,k);
					double const zdif = zcen0 - best_position(3,k);
					dist2 = vdw::atom_vdw(type0,type1) -
					 ( ( xdif * xdif ) + ( ydif * ydif ) + ( zdif * zdif ) );

					if ( dist2 > 0.0 ) bump_score += dist2;
				}
			}               // k
//---------
// check j-centroid to i-pentamer
//---------
			for ( int k = itemp; k <= itemp+4; ++k ) { // carol note itemp not jtemp
				type2 = atom_type(k); // vdw calc
				if ( type2 != 0 ) { // Not gly c-beta
					double const xdif = xcen3 - position(1,k);
					double const ydif = ycen3 - position(2,k);
					double const zdif = zcen3 - position(3,k);
					dist2 = vdw::atom_vdw(type3,type2) - // type3 is j centroid type
					 ( ( xdif * xdif ) + ( ydif * ydif ) + ( zdif * zdif ) );

					if ( dist2 > 0.0 ) {
						bump_score += dist2;
					}
				}
			}               // k


//------
// i-pentamer to j-pentamer
//------
			if ( alpha_alpha >= ca_dist_cutoff2 ) goto L100; // next j

			for ( int k = jtemp; k <= jtemp+4; ++k ) { // all backbone atoms in residue j
				type1 = atom_type(k);
				xpos1 = best_position(1,k);
				ypos1 = best_position(2,k);
				zpos1 = best_position(3,k);
				if ( type1 == 0 ) continue; // glycine c-beta; next k

				for ( int ii = itemp; ii <= itemp+4; ++ii ) {
				 // all backbone atoms in residue i
					type2 = atom_type(ii);
					if ( type2 != 0 ) { // Not glycine c-beta
						double const xdif = xpos1 - position(1,ii);
						double const ydif = ypos1 - position(2,ii);
						double const zdif = zpos1 - position(3,ii);
						dist2 = vdw::atom_vdw(type1,type2) -
						 ( ( xdif * xdif ) + ( ydif * ydif ) + ( zdif * zdif ) );

						if ( dist2 > 0.0 ) bump_score += dist2;
					}
				}            // end ii

			}               // end k
L100:;
		}                  // end j
	}                     // end i
	bump_score *= 0.8f; // required to keep on same scale as vdw_score

}



////////////////////////////////////////////////////////////////////////////////
/// @begin wobble_is_close_to_chainbreak
///
/// @brief
///   Screens regions around fragment insertion for chainbreaks in the fold tree...
///   don't bother to wobble if there's a chainbreak already.
///
/// @detailed
///
/// @param  frag_begin - [in/out]? -
/// @param  size - [in/out]? -
/// @param  nwobble - [in/out]? - wobble this many residues
/// @param  wobble_gap - [in/out]? - position of first wobbled residue
///        relative to frag end, -1 is first or last frag res, 0 is
///        0 is adjacent to frag, positive values are outside frag
/// @param  rama_score - [in/out]? -
/// @param  msd_score - [in/out]? -
/// @param  gfrag - [in/out]? -
/// @param  fold_begin - [in/out]? -
/// @param  fold_end - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///   rhiju
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
wobble_is_close_to_a_chainbreak(
	int & frag_begin,
	int size,
	int nwobble,
	int wobble_gap,
	float & rama_score,
	float & msd_score,
	bool & gfrag,
	int & fold_begin,
	int & fold_end
)
{
	if ( moves_check_current_pose() ) {
		// Yes, its a pose! There might be a chainbreak nearby.
		// Places that add_fast_wobble will look to "absorb" effect of fragment insertion.
		int const possible_wobble_begin = frag_begin - wobble_gap - nwobble - 1;
		int const possible_wobble_end   = frag_begin +size + wobble_gap + nwobble;

		pose_ns::Pose const & pose = moves_get_current_pose();
		int const nres = pose.total_residue();
		for (int i = possible_wobble_begin; i <= possible_wobble_end; i++){
			if (i >= 1 && i <= nres && pose.is_cutpoint(i)) { // no need to wobble!
				//				std::cout << "There's a chainbreak at: " << i << ". No need to wobble." << std::endl;
				rama_score = 0.0;
				msd_score = 0.0;
				gfrag = true;
				fold_begin = frag_begin;
				fold_end   = frag_begin+size;
				return true; // Found a chainbreak.
			}
		}
	}

	// No chainbreak found -- go ahead with the wobble
	return false;
}
